﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.BusinessLogic.CurveEditorParameters;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// Class for the view model of the EmitterEmissionVelocityViewModel.
    /// </summary>
    public class EmitterEmissionVelocityViewModel : PropertyGroupViewModel<EmitterEmissionVelocityData>, IModificationFlagOwner
    {
        /// <summary>
        /// 全方向初速アニメ値に負数を有効にするか
        /// </summary>
        private bool enabledOmnidirectionalVelocityMinusValue;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EmitterEmissionVelocityViewModel(
            HierarchyViewModel parent, EmitterEmissionVelocityData dataModel)
            : base(parent, dataModel)
        {
            this.EmitterOmniVelocityAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.OmnidirectionalVelocityAnimation, InitialKeyValues.OmniVelocity);
            this.Children.Add(this.EmitterOmniVelocityAnimationViewModel);
            this.EmitterOrientedDirectionalVelocityAnimationViewModel = new EmitterAnimationViewModel(
                this, dataModel.OrientedDirectionVelocityAnimation, InitialKeyValues.OrientedVelocity);
            this.Children.Add(this.EmitterOrientedDirectionalVelocityAnimationViewModel);

            this.OnShowCurveEditorDialog = new AnonymousExecutable((p) =>
            {
                if (string.Equals(p, "Omni"))
                {
                    this.UpdateOmniVelocityAnimation();
                }
                else if (string.Equals(p, "Oriented"))
                {
                    this.UpdateOrientedDirectionalVelocityAnimation();
                }

                WorkspaceRootViewModel.Instance.ShowCurveEditor();
            });

            // 全方向初速のエミッタ時間アニメの設定
            this.EmitterOmniVelocityAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsOmniVelocityCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterOmniVelocityAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableOmniVelocityAnimation);
                    this.UpdateOmniVelocityAnimation();
                }
            };
            this.EmitterOmniVelocityAnimationViewModel.SetDefaultSnapSetting("EmitterEmissionOmniVelocity");
            this.UpdateOmniVelocityAnimation();

            // 指定方向初速のエミッタ時間アニメの設定
            this.EmitterOrientedDirectionalVelocityAnimationViewModel.PropertyChanged += (s, e) =>
            {
                this.OnPropertyChanged(() => this.IsOrientedDirectionalVelocityCurveModified);
                if (IsRaisedProperty(e, () => this.EmitterOrientedDirectionalVelocityAnimationViewModel.EnableAnimation))
                {
                    this.OnPropertyChanged(() => this.EnableOrientedDirectionalVelocityAnimation);
                    this.UpdateOrientedDirectionalVelocityAnimation();
                }
            };
            this.EmitterOrientedDirectionalVelocityAnimationViewModel.SetDefaultSnapSetting("EmitterEmissionDirectionalVelocity");
            this.UpdateOrientedDirectionalVelocityAnimation();

            this.EnabledOmnidirectionalVelocityMinusValue = this.GetDataModelValue<float>("OmnidirectionalVelocity") < 0.0f;

            var mmfvm = new MultipleModificationFlagsViewModel(this);

            var emitterOmniVelocityParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.OmnidirectionalVelocity),
                ViewModelBase.NameOf(() => this.IsOmniVelocityCurveModified));
            mmfvm.SetPropertyDictionary("EmitterOmniVelocityParams", emitterOmniVelocityParamsProperties);
            mmfvm.SetModificationDictionary("EmitterOmniVelocityParams", new IModificationPropertyOwner[] { this.EmitterOmniVelocityAnimationViewModel });

            var emitterOrientedDirectionalVelocityParamsProperties = EnumerableUtility.Enumerate(
                ViewModelBase.NameOf(() => dataModel.OrientedVelocity),
                ViewModelBase.NameOf(() => this.IsOrientedDirectionalVelocityCurveModified));
            mmfvm.SetPropertyDictionary("EmitterOrientedDirectionalVelocityParams", emitterOrientedDirectionalVelocityParamsProperties);
            mmfvm.SetModificationDictionary("EmitterOrientedDirectionalVelocityParams", new IModificationPropertyOwner[] { this.EmitterOrientedDirectionalVelocityAnimationViewModel });

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            this.ModificationFlagViewModel = mmfvm;
        }

        /// <summary>
        /// 全方向初速のエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsOmniVelocityCurveModified
        {
            get { return this.EmitterOmniVelocityAnimationViewModel.IsModified; }
        }

        /// <summary>
        /// 指定方向初速のエミッタ時間アニメが変更されたか
        /// </summary>
        public bool IsOrientedDirectionalVelocityCurveModified
        {
            get { return this.EmitterOrientedDirectionalVelocityAnimationViewModel.IsModified;  }
        }

        /// <summary>
        /// 全方向初速アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableOmniVelocityAnimation
        {
            get
            {
                return this.EmitterOmniVelocityAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterOmniVelocityAnimationViewModel.EnableAnimation = value;
                this.UpdateOmniVelocityAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// 指定方向初速アニメの有効/無効を取得または設定します。
        /// </summary>
        public bool EnableOrientedDirectionalVelocityAnimation
        {
            get
            {
                return this.EmitterOrientedDirectionalVelocityAnimationViewModel.EnableAnimation;
            }

            set
            {
                this.EmitterOrientedDirectionalVelocityAnimationViewModel.EnableAnimation = value;
                this.UpdateOrientedDirectionalVelocityAnimation();
                if (value)
                {
                    WorkspaceRootViewModel.Instance.ShowCurveEditor();
                }
            }
        }

        /// <summary>
        /// 全方向初速アニメ値に負数を有効にするか
        /// </summary>
        public bool EnabledOmnidirectionalVelocityMinusValue
        {
            get { return this.enabledOmnidirectionalVelocityMinusValue; }
            set { this.SetValue(ref this.enabledOmnidirectionalVelocityMinusValue, value); }
        }

        /// <summary>
        /// EmitterOmniVelocityAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterOmniVelocityAnimationViewModel { get; private set; }

        /// <summary>
        /// EmitterOrientedDirectionalVelocityAnimationViewModel.
        /// </summary>
        public EmitterAnimationViewModel EmitterOrientedDirectionalVelocityAnimationViewModel { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// カーブエディタを表示するExecutableを取得します。
        /// </summary>
        public IExecutable OnShowCurveEditorDialog { get; private set; }

        /// <summary>
        /// ペースト時に全方向初速が負数だったらボタンの状態を更新する
        /// </summary>
        public override void FirePropertyChanges()
        {
            base.FirePropertyChanges();
            this.EnabledOmnidirectionalVelocityMinusValue = this.GetDataModelValue<float>("OmnidirectionalVelocity") < 0.0f;

            this.UpdateOmniVelocityAnimation();
            this.UpdateOrientedDirectionalVelocityAnimation();
        }

        /// <summary>
        /// Update child view models with the current data model.
        /// This method is usually called when data model is modified, thus some child
        /// view models might need to be created or removed according to the data model.
        /// </summary>
        public override void UpdateChildViewModels()
        {
            base.UpdateChildViewModels();
            this.EnabledOmnidirectionalVelocityMinusValue =
                this.GetDataModelValue<float>(() => this.DataModel.OmnidirectionalVelocity) < 0.0f;
        }

        /// <summary>
        /// 全方向初速アニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateOmniVelocityAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1.0f,
                LabelDigit = 2,
                MaxLimit = 1000000.0f,
                MinLimit = -1000000.0f,
                DefaultValue = InitialKeyValues.OmniVelocity[0],
                DefaultZeroPin = 3,
                Channels = new List<string>
                    {
                        "X",
                    },
                DataContext = this.EmitterOmniVelocityAnimationViewModel,
                AnimationName = "EmitterEmissionOmniVelocity",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }

        /// <summary>
        /// 指定方向初速アニメのエディタ接続更新
        /// </summary>
        /// <param name="forceDisconnect">強制的に切断する時はtrue.</param>
        public void UpdateOrientedDirectionalVelocityAnimation(bool forceDisconnect = false)
        {
            var editorParam = new EmitterAnimationEditorParameter
            {
                NormalizeAt = 1.0f,
                LabelDigit = 2,
                MaxLimit = 1000000.0f,
                MinLimit = 0.0f,
                DefaultValue = InitialKeyValues.OrientedVelocity[0],
                DefaultZeroPin = -1,
                Channels = new List<string>
                    {
                        "X",
                    },
                DataContext = this.EmitterOrientedDirectionalVelocityAnimationViewModel,
                AnimationName = "EmitterEmissionDirectiveVelocity",
            };

            if (!forceDisconnect)
            {
                WorkspaceRootViewModel.Instance.ConnectCurveEditor(editorParam);
            }
            else
            {
                WorkspaceRootViewModel.Instance.DisconnectCurveEditor(editorParam);
            }
        }
    }
}
