﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// EmitterCombinerEquationDataに対応するビューモデルです。
    /// </summary>
    public class EmitterCombinerEquationViewModel : PropertyGroupViewModel<EmitterCombinerEquationData>, IModificationFlagOwner
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        private readonly string[] ignorePropertyNames = new string[]
        {
            "LabelTexture0",
            "Texture1ExpressionVisibility"
        };

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">EmitterCombinerEquationData</param>
        public EmitterCombinerEquationViewModel(
            HierarchyViewModel parent, EmitterCombinerEquationData dataModel)
            : base(parent, dataModel)
        {
            this.CalcTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterCombinerOperatorMul, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterCombinerOperatorAdd, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterCombinerOperatorSub, 2),
            };

            this.ColorChannelTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorChannelTypeRGB, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorChannelTypeOneMinusRGB, 2),
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorChannelTypeAlpha, 3),
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorChannelTypeOneMinusAlpha, 4),
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorChannelTypeOne, 1),
            };

            this.AlphaChannelTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaChannelTypeA, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaChannelTypeOneMinusA, 3),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaChannelTypeR, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaChannelTypeOneMinusR, 4),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaChannelTypeOne, 2),
            };

            this.ColorExpressionTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorEquation0, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorEquation1, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorEquation2, 2),
                new KeyValuePair<string, object>(Properties.Resources.EmitterColorEquation3, 3),
            };

            this.AlphaExpressionTypeItems = new []
            {
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaEquation0, 0),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaEquation1, 1),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaEquation2, 2),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaEquation3, 3),
                new KeyValuePair<string, object>(Properties.Resources.EmitterAlphaEquation4, 4),
            };

            // カラーコンバイナのUIControlSetに関する設定を行なう.
            var colorCombinerProperties = EnumerableUtility.Enumerate(
                "ColorTexture0Channel",
                "ColorTexture1BlendType",
                "ColorTexture1Channel",
                "ColorTexture2BlendType",
                "ColorTexture2Channel",
                "ColorTextureExpression",
                "ColorPrimitiveExpression",
                "ColorPrimitiveChannel");

            // アルファコンバイナのUIControlSetに関する設定を行なう.
            var alphaCombinerProperties = EnumerableUtility.Enumerate(
                "AlphaTexture0Channel",
                "AlphaTexture1BlendType",
                "AlphaTexture1Channel",
                "AlphaTexture2BlendType",
                "AlphaTexture2Channel",
                "AlphaPrimitiveExpression",
                "AlphaPrimitiveChannel",
                "AlphaExpression");

            // Always create the modification flag view model IN THE END of the constructor
            // to prevent any initialization triggers the modification events.
            var multipleModificationFlagsViewModel = new MultipleModificationFlagsViewModel(this);

            multipleModificationFlagsViewModel.SetPropertyDictionary("ColorCombinerControlSet", colorCombinerProperties);
            multipleModificationFlagsViewModel.SetPropertyDictionary("AlphaCombinerControlSet", alphaCombinerProperties);

            multipleModificationFlagsViewModel.AddIgnoreProperties(this.ignorePropertyNames);

            this.ModificationFlagViewModel = multipleModificationFlagsViewModel;
        }

        /// <summary>
        /// テクスチャ0のラベルを取得します。
        /// </summary>
        public string LabelTexture0
        {
            get
            {
                int shaderType = this.GetShaderType();
                switch (shaderType)
                {
                    //// 通常
                    case 0:
                        return Properties.Resources.EmitterCombinerLabelTexture0Normal;
                    //// カラーバッファ屈折
                    case 1:
                        return Properties.Resources.EmitterCombinerLabelTexture0Refraction;
                    //// テクスチャ歪み
                    case 2:
                        return Properties.Resources.EmitterCombinerLabelTexture0Distortion;
                }

                return string.Empty;
            }

            set
            {
            }
        }

        /// <summary>
        /// テクスチャ1の演算式の表示On/Offを取得します。
        /// </summary>
        public bool Texture1ExpressionVisibility
        {
            get
            {
                int shaderType = this.GetShaderType();
                switch (shaderType)
                {
                    //// 通常: On
                    case 0:
                        return true;
                    //// カラーバッファ屈折: On
                    case 1:
                        return true;
                    //// テクスチャ歪み: Off
                    case 2:
                        return false;
                }

                return false;
            }

            set
            {
            }
        }

        /// <summary>
        /// 演算式の項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> CalcTypeItems { get; private set; }

        /// <summary>
        /// カラーチャンネルの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ColorChannelTypeItems { get; private set; }

        /// <summary>
        /// アルファチャンネルの項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AlphaChannelTypeItems { get; private set; }

        /// <summary>
        /// カラー計算式の項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> ColorExpressionTypeItems { get; private set; }

        /// <summary>
        /// アルファ計算式の項目を取得します.
        /// </summary>
        public IEnumerable<KeyValuePair<string, object>> AlphaExpressionTypeItems { get; private set; }

        /// <summary>
        /// Get the view model that holds the modification flags of
        /// this view model's properties.
        /// </summary>
        public ModificationFlagViewModel ModificationFlagViewModel { get; private set; }

        /// <summary>
        /// シェーダ設定が変更されたときの処理を行います。
        /// このメソッドのイベントハンドラへの登録は親ビューモデルで行います。
        /// </summary>
        /// <param name="sender">イベント発生源</param>
        /// <param name="e">イベント情報</param>
        public void OnShaderPropertyChanged(object sender, System.ComponentModel.PropertyChangedEventArgs e)
        {
            if (e.PropertyName == "ShaderType")
            {
                this.OnPropertyChanged(() => this.LabelTexture0);
                this.OnPropertyChanged(() => this.Texture1ExpressionVisibility);
            }
        }

        /// <summary>
        /// シェーダタイプを取得します。
        /// </summary>
        /// <returns>シェーダタイプを返します。</returns>
        private int GetShaderType()
        {
            var parent = ViewModelBase.GetParent<EmitterCombinerViewModel>(this);
            Debug.Assert(parent != null, "ビューモデルの構造が不正。");

            EmitterCombinerShaderViewModel shader = parent.EmitterCombinerShaderViewModel;
            Debug.Assert(shader != null, "ビューモデルの構造が不正。");

            int shaderType = shader.GetDataModelValue<int>(() => shader.ShaderType);
            return shaderType;
        }
    }
}
