﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;
using EffectMaker.UILogic.Attributes;
using EffectMaker.UILogic.Commands;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// カーブエディタを持つViewModelが必要なプロパティを集約したクラス
    /// </summary>
    public class CurveEditorOwnerViewModel : PropertyGroupViewModel
    {
        /// <summary>
        /// オプションに保存するスナップ設定名です。
        /// </summary>
        private string snapSettingName = string.Empty;

        /// <summary>
        /// EnableOtherKeySnap プロパティのバッキングフィールドです。
        /// </summary>
        private bool enableOtherKeySnap;

        /// <summary>
        /// SnapTime プロパティのバッキングフィールドです。
        /// </summary>
        private int snapTime;

        /// <summary>
        /// SnapValue プロパティのバッキングフィールドです。
        /// </summary>
        private int snapValue;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public CurveEditorOwnerViewModel(
            HierarchyViewModel parent, DataModelBase dataModel)
            : base(parent, dataModel)
        {
            this.OnResetExecutable = new AnonymousExecutable(this.OnReset);
            this.OnGetParticleLifeExecutable = new AnonymousExecutable((o) =>
            {
                var param = o as float[];
                var emitterVm = this.FindNearestParentOfType<EmitterViewModel>();
                param[0] = emitterVm.EmitterParticleViewModel.EmitterParticleLifeViewModel.GetDataModelValue<int>("Life");
            });

            this.InitSettings();
        }

        /// <summary>
        /// Get an executable to run when reset.
        /// </summary>
        public IExecutable OnResetExecutable { get; private set; }

        /// <summary>
        /// パーティクルの寿命を取得するExecutableを取得します。
        /// </summary>
        public IExecutable OnGetParticleLifeExecutable { get; private set; }

        /// <summary>
        /// 初期キーの値
        /// </summary>
        public Vector4f DefaultValue { get; set; }

        /// <summary>
        /// X軸のチェックボックス
        /// </summary>
        public bool EnableX { get; set; }

        /// <summary>
        /// Y軸のチェックボックス
        /// </summary>
        public bool EnableY { get; set; }

        /// <summary>
        /// W軸のチェックボックス
        /// </summary>
        public bool EnableZ { get; set; }

        /// <summary>
        /// Z軸のチェックボックス
        /// </summary>
        public bool EnableW { get; set; }

        /// <summary>
        /// 他のキーにスナップするモードのボタン
        /// </summary>
        public bool EnableOtherKeySnap
        {
            get
            {
                return this.enableOtherKeySnap;
            }

            set
            {
                this.enableOtherKeySnap = value;
                this.UpdateSnapSettingOption();
            }
        }

        /// <summary>
        /// 時間スナップ
        /// </summary>
        public int SnapTime
        {
            get
            {
                return this.snapTime;
            }

            set
            {
                this.snapTime = value;
                this.UpdateSnapSettingOption();
            }
        }

        /// <summary>
        /// 値スナップ
        /// </summary>
        public int SnapValue
        {
            get
            {
                return this.snapValue;
            }

            set
            {
                this.snapValue = value;
                this.UpdateSnapSettingOption();
            }
        }

        /// <summary>
        /// 0固定モード
        /// </summary>
        public int ZeroPinMode { get; set; }

        /// <summary>
        /// Gets the interpolation mode.
        /// </summary>
        [UseDataModelOriginalValue]
        public AnimationTableData AnimationTable
        {
            get
            {
                return this.GetDataModelValue(() => this.AnimationTable);
            }

            set
            {
                int count = this.AnimationTable.Count;
                var binder = new EffectMakerSetMemberBinder("AnimationTable", false, false);
                MessageBlockerWithSendBinaryOnce.ExecuteOnDemandReload(
                    () => this.TrySetMember(binder, value),
                    value.Count != count,
                    (DataModelBase)this.Proxy.DataModel);
            }
        }

        /// <summary>
        /// スナップ設定のデフォルト値を設定します。
        /// </summary>
        /// <param name="settingName">設定名</param>
        /// <param name="isEnabled">スナップを有効にするかどうか</param>
        /// <param name="snapTime">時間スナップ</param>
        /// <param name="snapValue">値スナップ</param>
        public void SetDefaultSnapSetting(string settingName, bool enableSnap = true, int snapTime = 0, int snapValue = 2)
        {
            // スナップのデフォルト値を適用
            this.EnableOtherKeySnap = enableSnap;
            this.SnapTime = snapTime;
            this.SnapValue = snapValue;

            // オプションに保存されたデフォルト値を取得
            CurveEditorSnapSetting setting;
            OptionStore.RootOptions.Interface.CurveEditorSnapSettings.TryGetValue(settingName, out setting);

            // オプションに保存されたデフォルト値があればそれを適用する
            if (setting != null)
            {
                this.EnableOtherKeySnap = setting.EnableSnap;
                this.SnapTime = setting.SnapValueX;
                this.SnapValue = setting.SnapValueY;
            }

            // スナップ設定名に有効な値をセットすると、スナップ値の変更がオプションに反映されるようになるため
            // 全てのスナップ値の変更が終わってからスナップ設定名をセットする
            this.snapSettingName = settingName;
        }

        /// <summary>
        /// 現在のスナップ設定をオプションに反映させます。
        /// </summary>
        private void UpdateSnapSettingOption()
        {
            if (string.IsNullOrEmpty(this.snapSettingName) == false)
            {
                CurveEditorSnapSetting setting = new CurveEditorSnapSetting
                {
                    EnableSnap = this.EnableOtherKeySnap,
                    SnapValueX = this.SnapTime,
                    SnapValueY = this.SnapValue
                };

                OptionStore.RootOptions.Interface.CurveEditorSnapSettings[this.snapSettingName] = setting;
            }
        }

        /// <summary>
        /// Gets or sets a value indicating whether reset settings.
        /// </summary>
        public void OnReset()
        {
            if (ExportableViewModel.IsPasting)
            {
                return;
            }

            using (new MessageBlockerWithSendBinaryOnce(this.Proxy.DataModel))
            using (new CommandCombiner())
            {
                this.TrySetMember(new EffectMakerSetMemberBinder("LoopMode", false, false), 0);
                this.TrySetMember(new EffectMakerSetMemberBinder("Randomize", false, false), 0);
                this.SetDataModelValue(100, "LoopEnd");
                this.InitData();
            }

            this.InitSettings();
        }

        /// <summary>
        /// 値以外の設定を初期化
        /// </summary>
        protected void InitSettings()
        {
            this.EnableX = true;
            this.EnableY = true;
            this.EnableZ = true;
            this.EnableW = true;
            this.EnableOtherKeySnap = true;
            this.SnapTime = 0;
            this.SnapValue = 2;
            this.ZeroPinMode = -2;
        }

        /// <summary>
        /// 値を初期化
        /// </summary>
        protected void InitData()
        {
            var table = new AnimationTableData(
                0,
                this.DefaultValue.X,
                this.DefaultValue.Y,
                this.DefaultValue.Z,
                this.DefaultValue.W);
            this.SetDataModelValue(table, () => this.AnimationTable);
        }
    }

    /// <summary>
    /// ジェネリック版のCurveEditorOwnerViewModelです。
    /// </summary>
    /// <typeparam name="T">データモデルの型</typeparam>
    public abstract class CurveEditorOwnerViewModel<T> : CurveEditorOwnerViewModel where T : DataModelBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">関連付けるデータモデル</param>
        protected CurveEditorOwnerViewModel(HierarchyViewModel parent, T dataModel)
            : base(parent, dataModel)
        {
        }

        /// <summary>
        /// 関連付けられたデータモデルを取得します。
        /// </summary>
        public T DataModel
        {
            get { return this.Proxy.DataModel; }
        }
    }
}
