﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Disposables;
using EffectMaker.Foundation.Editting;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.UILogic.ViewModels
{
    /// <summary>
    /// カラーピッカーを持つViewModelが必要なプロパティを集約したクラス
    /// </summary>
    public class ColorPickerOwnerViewModel : PropertyGroupViewModel
    {
        /// <summary>
        /// 変更の対象に含めないプロパティ名のリストです.
        /// </summary>
        protected readonly string[] IgnoreColorPickerPropertyNames = new string[]
        {
            "EditingColorInfo",
            "ColorPickerActive",
            "ColorPickerRgbEditEnabled",
            "ColorPickerAlphaEditEnabled",
            "ColorPickerValue",
            "ColorPickerTime",
            "ColorPickerTimeEditEnabled",
            "ColorPickerEditingValue",
            "ColorPickerEditingTime",
            "ColorPickerLockingTabOrder"
        };

        /// <summary>
        /// 色編集情報
        /// </summary>
        private EditingColorInfo editingColorInfo = new EditingColorInfo();

        /// <summary>
        /// カラーピッカーの値
        /// </summary>
        private ColorRgba colorPickerValue = new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);

        /// <summary>
        /// カラーピッカーの編集中値
        /// </summary>
        private ColorRgba colorPickerEditingValue = new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);

        /// <summary>
        /// カラーピッカーの時間
        /// </summary>
        private int colorPickerTime;

        /// <summary>
        /// カラーピッカーの編集中時間
        /// </summary>
        private int colorPickerEditingTime;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public ColorPickerOwnerViewModel(
            HierarchyViewModel parent, DataModelBase dataModel)
            : base(parent, dataModel)
        {
            this.BlockingFunc = () => false;
        }

        /// <summary>
        /// 色編集情報
        /// </summary>
        public EditingColorInfo EditingColorInfo
        {
            get
            {
                return this.editingColorInfo;
            }

            set
            {
                if (value == null)
                {
                    return;
                }

                if (this.IsBlocking)
                {
                    this.DisableColorPicker();
                    return;
                }

                this.SetValue(ref this.editingColorInfo, new EditingColorInfo(value));

                this.OnPropertyChanged(() => this.ColorPickerActive);
                this.OnPropertyChanged(() => this.ColorPickerRgbEditEnabled);
                this.OnPropertyChanged(() => this.ColorPickerAlphaEditEnabled);
                this.OnPropertyChanged(() => this.ColorPickerTimeEditEnabled);
                this.OnPropertyChanged(() => this.ColorPickerLockingTabOrder);

                this.ColorPickerValue = EditingColorInfo.Color;
                this.ColorPickerTime = EditingColorInfo.Time;

                // エミッタを切り替えてデフォルト値のカラーを選択したときにカラーピッカーの値が
                // 前選択したカラーのままになるため強制的にカラーを更新する
                this.OnPropertyChanged(() => this.ColorPickerValue);
                this.OnPropertyChanged(() => this.ColorPickerTime);
            }
        }

        /// <summary>
        /// アクティブなカラーピッカーか？
        /// </summary>
        public bool ColorPickerActive
        {
            get { return this.editingColorInfo.Control != null; }

            set { }
        }

        /// <summary>
        /// RGB編集を有効にするか？
        /// </summary>
        public bool ColorPickerRgbEditEnabled
        {
            get { return this.editingColorInfo.RgbEditEnabled; }

            set { }
        }

        /// <summary>
        /// アルファ編集を有効にするか？
        /// </summary>
        public bool ColorPickerAlphaEditEnabled
        {
            get { return this.editingColorInfo.AlphaEditEnabled; }

            set { }
        }

        /// <summary>
        /// カラーピッカーの値
        /// </summary>
        public ColorRgba ColorPickerValue
        {
            get
            {
                return this.colorPickerValue;
            }

            set
            {
                if (this.IsBlocking)
                {
                    return;
                }

                this.SetValue(ref this.colorPickerValue, value);

                this.EditingColorInfo.Color = value;
                this.OnPropertyChanged(() => this.EditingColorInfo);
            }
        }

        /// <summary>
        /// カラーピッカーの時間
        /// </summary>
        public int ColorPickerTime
        {
            get
            {
                return this.colorPickerTime;
            }

            set
            {
                if (this.IsBlocking)
                {
                    return;
                }

                this.SetValue(ref this.colorPickerTime, value);

                this.EditingColorInfo.Time = value;
                this.OnPropertyChanged(() => this.EditingColorInfo);
            }
        }

        /// <summary>
        /// カラーピッカーの時間を有効にするか？
        /// </summary>
        public bool ColorPickerTimeEditEnabled
        {
            get { return this.editingColorInfo.TimeEditEnabled; }
        }

        /// <summary>
        /// カラーピッカーの編集中値
        /// </summary>
        public ColorRgba ColorPickerEditingValue
        {
            get
            {
                return this.colorPickerEditingValue;
            }

            set
            {
                if (this.IsBlocking)
                {
                    return;
                }

                this.SetValue(ref this.colorPickerEditingValue, value);

                using (new AnonymousDisposable(() => this.EditingColorInfo.IsEditing = false))
                {
                    this.EditingColorInfo.IsEditing = true;
                    this.EditingColorInfo.Color = value;
                    this.OnPropertyChanged(() => this.EditingColorInfo);
                }
            }
        }

        /// <summary>
        /// カラーピッカーの編集中時間
        /// </summary>
        public int ColorPickerEditingTime
        {
            get
            {
                return this.colorPickerEditingTime;
            }

            set
            {
                if (this.IsBlocking)
                {
                    return;
                }

                this.SetValue(ref this.colorPickerEditingTime, value);
                this.OnPropertyChanged(() => this.EditingColorInfo);
            }
        }

        /// <summary>
        /// タブオーダーがカラーピッカー内でロックされているか否かを返します。
        /// </summary>
        public bool ColorPickerLockingTabOrder
        {
            get { return this.editingColorInfo.LockingTabOrder; }
        }

        /// <summary>
        /// カラーピッカーの処理をブロックするかどうかを返します。
        /// </summary>
        protected bool IsBlocking
        {
            get
            {
                return ExportableViewModel.IsPasting || this.BlockingFunc();
            }
        }

        /// <summary>
        /// カラーピッカーの処理をブロックする評価関数を取得・設定します。
        /// </summary>
        protected Func<bool> BlockingFunc { get; set; }

        /// <summary>
        /// Resend PropertyChanged notification for all properties.
        /// This is required when the data model changes independently from the view model.
        /// </summary>
        public override void FirePropertyChanges()
        {
            foreach (string propertyName in this.EnumerateViewModelPropertyNames())
            {
                if (propertyName != "EditingColorInfo")
                {
                    this.OnPropertyChanged(propertyName);
                }
            }

            foreach (string propertyName in this.Proxy.GetDynamicMemberNames())
            {
                this.OnPropertyChanged(propertyName);
            }

            // Call FirePropertyChanges method for the children.
            this.FirePropertyChangeEventForChildren();
        }

        /// <summary>
        /// カラーピッカーを無効にする
        /// </summary>
        protected void DisableColorPicker()
        {
            this.editingColorInfo.Control = null;
            this.OnPropertyChanged(() => this.ColorPickerActive);
        }

        /// <summary>
        /// カラーピッカーを更新する
        /// </summary>
        /// <param name="targetName">ターゲット名.</param>
        /// <param name="color">カラー.</param>
        /// <param name="time">時間.</param>
        protected void RefreshColorPicker(string targetName, ColorRgba color, int time = 0)
        {
            if (this.IsBlocking)
            {
                return;
            }

            using (new AnonymousDisposable(() => this.EditingColorInfo.RefreshTargetName = null))
            {
                this.editingColorInfo.RefreshTargetName = targetName;
                this.editingColorInfo.RefreshColorPickerRequest = false;

                this.OnPropertyChanged(() => this.EditingColorInfo);

                // バインド先でカラーピッカー更新要求に答える
                if (this.editingColorInfo.RefreshColorPickerRequest)
                {
                    this.ColorPickerEditingValue = color;
                    this.ColorPickerEditingTime = time;
                }
            }
        }

        /// <summary>
        /// カラーピッカーを更新する(alpha)
        /// </summary>
        /// <param name="targetName">ターゲット名.</param>
        /// <param name="alpha">アルファ.</param>
        /// <param name="time">時間.</param>
        protected void RefreshColorPicker(string targetName, float alpha, int time = 0)
        {
            this.RefreshColorPicker(targetName, new ColorRgba(1.0f, 1.0f, 1.0f, alpha), time);
        }
    }

    /// <summary>
    /// ジェネリック版のColorPickerOwnerViewModelです。
    /// </summary>
    /// <typeparam name="T">データモデルの型</typeparam>
    public abstract class ColorPickerOwnerViewModel<T> : ColorPickerOwnerViewModel where T : DataModelBase
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="parent">親ビューモデル</param>
        /// <param name="dataModel">関連付けるデータモデル</param>
        protected ColorPickerOwnerViewModel(HierarchyViewModel parent, T dataModel)
            : base(parent, dataModel)
        {
        }

        /// <summary>
        /// 関連付けられたデータモデルを取得します。
        /// </summary>
        public T DataModel
        {
            get { return this.Proxy.DataModel; }
        }
    }
}
