﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Extensions;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// データモデルの内容をデフォルト値にリセットするコマンドです。
    /// </summary>
    [Alias("ResetDefaultValue")]
    public class ResetDefaultValueCommand : CommandBase
    {
        /// <summary>
        /// コマンドスタックを持つターゲット
        /// </summary>
        private object commandStackTarget;

        /// <summary>
        /// 処理対象のビューモデル
        /// </summary>
        private HierarchyViewModel viewModel;

        /// <summary>
        /// リセット前のデータモデルのコピー
        /// </summary>
        private DataModelBase currentDataModel;

        /// <summary>
        /// 複数選択時の編集操作をアクションリストで保持します。
        /// </summary>
        private List<Action> multiUndoActions, multiRedoActions;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ResetDefaultValueCommand()
        {
            this.multiUndoActions = new List<Action>();
            this.multiRedoActions = new List<Action>();
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="viewModel">リセットするビューモデル</param>
        public ResetDefaultValueCommand(HierarchyViewModel viewModel)
            : this()
        {
            object[] args = new object[] { viewModel };

            this.Initialize(args);
        }

        /// <summary>
        /// コマンドを初期化します。
        /// </summary>
        /// <param name="args">コマンド引数</param>
        /// <returns>処理が成功したときはtrue、失敗したときはfalseを返します。</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || args.Length != 1)
            {
                return false;
            }

            this.viewModel = args[0] as HierarchyViewModel;

            if (this.viewModel == null)
            {
                return false;
            }

            // カスタムシェーダの基本グループはタブリセットと同じ扱いにする
            if (this.viewModel is EmitterCustomShaderBasicViewModel)
            {
                // できれば他の場所で書くべきだとは思う……
                this.viewModel = this.viewModel.Parent as HierarchyViewModel;
            }

            // アンドゥ処理のため、現在のデータモデルの状態をコピーする
            using (new EnableDataModelCloneSetter())
            {
                this.currentDataModel = (DataModelBase)this.viewModel.Proxy.DataModel.Clone();
            }

            // データモデルのデフォルト値が登録されているかチェック
            if (this.viewModel.Proxy.GetDefaultDataModel() == null)
            {
                return false;
            }

            // コマンドスタックを取得
            this.commandStackTarget = CommandBase.FindCommandStackTargetFromHierarchyObject(this.viewModel);

            // 初期化完了！
            this.IsInitialized = true;

            return true;
        }

        /// <summary>
        /// コマンドスタックを持つターゲットを取得します。
        /// </summary>
        /// <returns>ターゲットを返します。</returns>
        public override object GetTarget()
        {
            return this.commandStackTarget;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>処理が成功したときはtrue、失敗したときはfalseを返します。</returns>
        public override bool Do()
        {
            var values = this.viewModel.Proxy.GetDefaultDataModel();

            this.multiUndoActions.Clear();
            this.multiRedoActions.Clear();

            // 複数選択対象の同じプロパティに対して同じ値をセットする
            foreach (var node in WorkspaceRootViewModel.Instance.MultiSelectedNodes)
            {
                var sameTarget = HierarchyObjectExtensions.FindChildrenRecursive(
                    this.viewModel,
                    node,
                    MultiNodeEditUtil.IsSameTarget) as HierarchyViewModel;
                if (sameTarget != null)
                {
                    object currentValues = null;
                    using (new EnableDataModelCloneSetter())
                    {
                        currentValues = (DataModelBase)sameTarget.Proxy.DataModel.Clone();
                    }

                    Action undoAction = () => ResetProc(sameTarget, currentValues);
                    this.multiUndoActions.Add(undoAction);

                    Action redoAction = () => ResetProc(sameTarget, values);
                    this.multiRedoActions.Add(redoAction);

                    redoAction();  // Do()の代用としてRedo()を実行
                }
            }

            // データモデルをデフォルトにリセット
            bool result =  ResetProc(this.viewModel, values);

            if (result)
            {
                // Update UI tool bar states.
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // バイナリデータを送信する
                using (new ForceRefreshBinary())
                {
                    ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.Proxy.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// コマンドの処理を元に戻します。
        /// </summary>
        /// <returns>処理が成功したときはtrue、失敗したときはfalseを返します。</returns>
        public override bool Undo()
        {
            // アクションリストに保存された同時編集をUndo
            foreach (var editAction in this.multiUndoActions)
            {
                try
                {
                    editAction();
                }
                catch (Exception e)
                {
                    // エミッタセットを削除された場合には対応しきれないのでログだけ吐いておく
                    Logger.Log(LogLevels.Error, "Multi Edit Undo Error:" + e.Message);
                }
            }

            // データモデルをリセット前の状態に戻す
            bool result =  ResetProc(this.viewModel, this.currentDataModel);

            if (result)
            {
                // Update UI tool bar states.
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // バイナリデータを送信する
                using (new ForceRefreshBinary())
                {
                    ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.Proxy.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// コマンドの処理をやり直します。
        /// </summary>
        /// <returns>処理が成功したときはtrue、失敗したときはfalseを返します。</returns>
        public override bool Redo()
        {
            var values = this.viewModel.Proxy.GetDefaultDataModel();

            // アクションリストに保存された同時編集をRedo
            foreach (var editAction in this.multiRedoActions)
            {
                try
                {
                    editAction();
                }
                catch (Exception e)
                {
                    // エミッタセットを削除された場合には対応しきれないのでログだけ吐いておく
                    Logger.Log(LogLevels.Error, "Multi Edit Redo Error:" + e.Message);
                }
            }

            // データモデルをデフォルトにリセット
            bool result =  ResetProc(this.viewModel, values);

            if (result)
            {
                // Update UI tool bar states.
                WorkspaceRootViewModel.Instance.UpdateUIStates();

                // バイナリデータを送信する
                using (new ForceRefreshBinary())
                {
                    ViewerMessageHelper.FindPropertyParentDataAndSend(this.viewModel.Proxy.DataModel);
                }
            }

            return result;
        }

        /// <summary>
        /// リセット処理のコアです。
        /// </summary>
        /// <param name="target">リセット対象のページビューモデル</param>
        /// <param name="values">値</param>
        /// <returns>処理が成功したときはtrue、失敗したときはfalseを返します。</returns>
        private static bool ResetProc(HierarchyViewModel target, object values)
        {
            try
            {
                ExportableViewModel.IsPasting = true;

                // 対象のデータモデルに値をセット
                DataModelBase dataModel = target.Proxy.DataModel;
                using (new EnableDataModelCloneSetter())
                {
                    dataModel.SetWithoutGuid(values);
                }

                using (new MessageDropSection())
                {
                    // Update the child view models with the new data model.
                    target.UpdateChildViewModels();

                    // Notify the UI that the values had changed.
                    target.FirePropertyChanges();
                }

                ExportableViewModel.IsPasting = false;

                // カーブエディタウィンドウの状態を更新
                WorkspaceRootViewModel.Instance.UpdateCurveEditorWindow(target);

                return true;
            }
            catch
            {
                ExportableViewModel.IsPasting = false;

                return false;
            }
        }
    }
}
