﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// 複数ノード同時編集コマンドで使用する関数群を集約したクラスです。
    /// </summary>
    public static class MultiNodeEditUtil
    {
        /// <summary>
        /// スタティックコンストラクタ
        /// </summary>
        static MultiNodeEditUtil()
        {
            DisableMultiNodeEdit = false;
        }

        /// <summary>
        /// 複数ノード同時編集を抑制するかどうかを取得または設定します。
        /// </summary>
        public static bool DisableMultiNodeEdit { get; set; }

        /// <summary>
        /// ビューモデルがownerと同階層のものかどうかを判定します。
        /// </summary>
        /// <param name="src">ソースビューモデル</param>
        /// <param name="obj">判定対象オブジェクト</param>
        /// <returns>一致していたらtrue,そうでなければfalse.</returns>
        public static bool IsSameTarget(object src, object obj)
        {
            var srcViewModel = src as HierarchyViewModel;
            var dataModelOwner = obj as HierarchyViewModel;
            if (srcViewModel == null || dataModelOwner == null)
            {
                return false;
            }

            // ビューモデルの型が、自分自身と親で一致していなければ同階層ではない
            if (srcViewModel.GetType() == dataModelOwner.GetType() &&
                srcViewModel.Parent != null && dataModelOwner.Parent != null &&
                srcViewModel.Parent.GetType() == dataModelOwner.Parent.GetType())
            {
                // テクスチャに関しては型だけだと判断付かないので、判定処理に委譲
                var srcTexVm = srcViewModel.Parent as EmitterTextureViewModel;
                var dstTexVm = dataModelOwner.Parent as EmitterTextureViewModel;
                if (srcTexVm != null && dstTexVm != null)
                {
                    return srcTexVm.IsSameTabNumber(dstTexVm);
                }

                // データモデルについても自分自身と親について同様にチェックする
                var srcDm = (DataModelBase)srcViewModel.Proxy.DataModel;
                var ownerDm = (DataModelBase)dataModelOwner.Proxy.DataModel;
                return srcDm.GetType() == ownerDm.GetType() &&
                       srcDm.Parent != null && ownerDm.Parent != null &&
                       srcDm.Parent.GetType() == ownerDm.Parent.GetType();
            }

            return false;
        }

        /// <summary>
        /// 値のセット処理を実行してリロードを検出します。
        /// </summary>
        public class ReloadDetector : IDisposable
        {
            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="setValueFunction">値のセット処理</param>
            public ReloadDetector(Func<bool> setValueFunction)
            {
                this.SendBinaryRequested = false;
                SendModifiedBinaryContext.SendBinaryRequested += this.SendBinaryRequestedNotify;
                this.Result = setValueFunction();
            }

            /// <summary>
            /// 値のセット処理結果を取得します。
            /// </summary>
            public bool Result { get; private set; }

            /// <summary>
            /// リロードが発生したかどうかを取得します。
            /// </summary>
            public bool SendBinaryRequested { get; private set; }

            /// <summary>
            /// ディスポーザーでイベント登録を解除し、通信状態を復帰します。
            /// </summary>
            public void Dispose()
            {
                SendModifiedBinaryContext.SendBinaryRequested -= this.SendBinaryRequestedNotify;

                if (this.SendBinaryRequested)
                {
                    ViewerMessageHelper.EndDisableBinaryConvertSection();
                }
            }

            /// <summary>
            /// バイナリリロードが発生しそうになったことを通知し、通信をブロックします。
            /// </summary>
            /// <param name="sender"></param>
            /// <param name="args"></param>
            private void SendBinaryRequestedNotify(object sender, EventArgs args)
            {
                this.SendBinaryRequested = true;
                ViewerMessageHelper.BeginDisableBinaryConvertSection();
                SendModifiedBinaryContext.SendBinaryRequested -= this.SendBinaryRequestedNotify;
            }
        }

        /// <summary>
        /// 複数ノード編集時のビット演算を管理するクラスです。
        /// </summary>
        public class BitOperator
        {
            /// <summary>
            /// 操作するビットを表すマスク値です。
            /// </summary>
            private readonly object bitMask;

            /// <summary>
            /// ビットを上げる操作か下げる操作かを表します。
            /// </summary>
            private readonly bool bitValue;

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="originalValue">主ノードの編集前値</param>
            /// <param name="newValue">主ノードの編集後値</param>
            public BitOperator(object originalValue, object newValue)
            {
                // 操作対象の型ごとに実装しています……いい手があったら置き換えてください
                if (newValue is ushort)
                {
                    ushort tmpOld = (ushort)originalValue;
                    ushort tmpNew = (ushort)newValue;
                    ushort tmpMask = (ushort)(tmpOld ^ tmpNew);
                    this.bitValue = (tmpNew & tmpMask) != 0;
                    this.bitMask = tmpMask;
                    return;
                }

                if (newValue is uint)
                {
                    uint tmpOld = (uint)originalValue;
                    uint tmpNew = (uint)newValue;
                    uint tmpMask = tmpOld ^ tmpNew;
                    this.bitValue = (tmpNew & tmpMask) != 0;
                    this.bitMask = tmpMask;
                    return;
                }

                if (newValue is ulong)
                {
                    ulong tmpOld = (ulong)originalValue;
                    ulong tmpNew = (ulong)newValue;
                    ulong tmpMask = tmpOld ^ tmpNew;
                    this.bitValue = (tmpNew & tmpMask) != 0;
                    this.bitMask = tmpMask;
                    return;
                }

                throw new InvalidCastException();
            }

            /// <summary>
            /// 主ノードと同じビット操作を行った結果を返します。
            /// </summary>
            /// <param name="baseValue">複ノードの編集前値</param>
            /// <returns>ビット操作適用後の値</returns>
            public object GetOperatedValue(object baseValue)
            {
                // 操作対象の型ごとに実装しています……いい手があったら置き換えてください
                if (baseValue is ushort)
                {
                    ushort tmpOld = (ushort)baseValue;
                    ushort tmpMask = (ushort)this.bitMask;
                    return (ushort)(this.bitValue ? tmpOld | tmpMask : tmpOld & ~tmpMask);
                }

                if (baseValue is uint)
                {
                    uint tmpOld = (uint)baseValue;
                    uint tmpMask = (uint)this.bitMask;
                    return this.bitValue ? tmpOld | tmpMask : tmpOld & ~tmpMask;
                }

                if (baseValue is ulong)
                {
                    ulong tmpOld = (ulong)baseValue;
                    ulong tmpMask = (ulong)this.bitMask;
                    return this.bitValue ? tmpOld | tmpMask : tmpOld & ~tmpMask;
                }

                throw new InvalidCastException();
            }
        }
    }
}
