﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Command;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.UILogic.Commands
{
    /// <summary>
    /// LoadEmitterSetCommand.
    /// </summary>
    [Alias("LoadEmitterSet")]
    public class LoadEmitterSetCommand : CommandBase
    {
        /// <summary>
        /// ワークスペースのビューモデルです.
        /// </summary>
        private WorkspaceViewModel workspaceViewModel;

        /// <summary>
        /// エミッタセットのファイルパスです.
        /// </summary>
        private string filePath;

        /// <summary>
        /// プレビューをエミッタセットに追加するかどうか
        /// </summary>
        private bool addPreview;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public LoadEmitterSetCommand()
        {
            this.CanUndo = false;
        }

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        /// <param name="workspaceViewModel">ワークスペースのビューモデル</param>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="addPreview">プレビューをエミッタセットに追加するかどうか</param>
        public LoadEmitterSetCommand(WorkspaceViewModel workspaceViewModel, string filePath, bool addPreview = true)
        {
            object[] args = new object[] { workspaceViewModel, filePath, addPreview };

            this.Initialize(args);
        }

        /// <summary>
        /// 読み込み時のノード選択を抑制するフラグ
        /// </summary>
        public static bool SuppressSelectionNode { get; set; }

        /// <summary>
        /// Initialize the command and process the arguments.
        /// </summary>
        /// <param name="args">The arguments for the command.</param>
        /// <returns>True on success.</returns>
        public override bool Initialize(object[] args)
        {
            if (args == null || !(args.Length == 2 || args.Length == 3))
            {
                return false;
            }

            // ワークスペースのビューモデル
            this.workspaceViewModel = args[0] as WorkspaceViewModel;

            if (this.workspaceViewModel == null)
            {
                return false;
            }

            // ファイルパス
            this.filePath = args[1] as string;

            if (string.IsNullOrEmpty(this.filePath))
            {
                return false;
            }

            // プレビューをエミッタセットに追加するかどうか
            if (args.Length > 2)
            {
                if (args[2] is bool == false)
                {
                    return false;
                }

                this.addPreview = (bool)args[2];
            }
            else
            {
                this.addPreview = true;
            }

            this.IsInitialized = true;
            return true;
        }

        /// <summary>
        /// ターゲットを取得します.
        /// </summary>
        /// <returns>ターゲットのオブジェクトインスタンス</returns>
        public override object GetTarget()
        {
            return null;
        }

        /// <summary>
        /// Execute the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Do()
        {
            LoadEmitterSetResultWithData loadEmitterSetResultWithData =
                this.workspaceViewModel.LoadEmitterSetFile(this.filePath, this.addPreview);
            EmitterSetViewModel emitterSetViewModel = loadEmitterSetResultWithData.ViewModel;

            if (emitterSetViewModel == null)
            {
                return false;
            }

            if (this.addPreview)
            {
                ViewerMessageHelper.SendEmitterSet(emitterSetViewModel.DataModel);
            }

            if (!SuppressSelectionNode)
            {
                // 読み込んだエミッタセットを選択状態に設定。
                emitterSetViewModel.IsSelected = true;
            }

            return true;
        }

        /// <summary>
        /// Undo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Undo()
        {
            return false;
        }

        /// <summary>
        /// Redo the command.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Redo()
        {
            return false;
        }
    }
}
