﻿// ---------------------------------------c-----------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Threading.Tasks;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using nw.g3d.nw4f_3dif;
using G3dIfLib;

using G3dTextureData = nw.g3d.iflib.TextureData;

namespace EffectMaker.TextureManager.Loader
{
    /// <summary>
    /// FTX テクスチャローダです。
    /// </summary>
    internal class FtxTextureLoader : ITextureLoader
    {
        /// <summary>
        /// Component selector変換テーブル
        /// </summary>
        private static readonly Dictionary<texture_info_comp_selValue, ColorComponents> ComponentSelectorConvertTable = new Dictionary<texture_info_comp_selValue, ColorComponents>
        {
            { texture_info_comp_selValue.r, ColorComponents.Red },
            { texture_info_comp_selValue.g, ColorComponents.Green },
            { texture_info_comp_selValue.b, ColorComponents.Blue },
            { texture_info_comp_selValue.a, ColorComponents.Alpha },
            { texture_info_comp_selValue.Item0, ColorComponents.Item0 },
            { texture_info_comp_selValue.Item1, ColorComponents.Item1 },
        };

        /// <summary>
        /// テクスチャタイプ変換テーブル
        /// </summary>
        private static readonly Dictionary<texture_info_dimensionType, TextureTypes> TextureTypeConvertTable = new Dictionary<texture_info_dimensionType, TextureTypes>
        {
            { texture_info_dimensionType.Item1d, TextureTypes.Dim1 },
            { texture_info_dimensionType.Item2d, TextureTypes.Dim2 },
            { texture_info_dimensionType.Item3d, TextureTypes.Dim3 },
            { texture_info_dimensionType.cube, TextureTypes.Cube },
            { texture_info_dimensionType.Item1d_array, TextureTypes.ArrayDim1 },
            { texture_info_dimensionType.Item2d_array, TextureTypes.ArrayDim2 },
            { texture_info_dimensionType.cube_array, TextureTypes.ArrayCube },
        };

        /// <summary>
        /// ピクセルフォーマット変換テーブル
        /// </summary>
        private static readonly Dictionary<texture_info_quantize_typeType, PixelFormats> PixelFormatConvertTable = new Dictionary<texture_info_quantize_typeType, PixelFormats>
        {
            { texture_info_quantize_typeType.unorm_8, PixelFormats.Unorm_8 },
            { texture_info_quantize_typeType.uint_8, PixelFormats.Uint_8 },
            { texture_info_quantize_typeType.snorm_8, PixelFormats.Snorm_8 },
            { texture_info_quantize_typeType.sint_8, PixelFormats.Sint_8 },
            { texture_info_quantize_typeType.unorm_4_4, PixelFormats.Unorm_4_4 },
            { texture_info_quantize_typeType.unorm_16, PixelFormats.Unorm_16 },
            { texture_info_quantize_typeType.uint_16, PixelFormats.Uint_16 },
            { texture_info_quantize_typeType.snorm_16, PixelFormats.Snorm_16 },
            { texture_info_quantize_typeType.sint_16, PixelFormats.Sint_16 },
            { texture_info_quantize_typeType.float_16, PixelFormats.Float_16 },
            { texture_info_quantize_typeType.unorm_8_8, PixelFormats.Unorm_8_8 },
            { texture_info_quantize_typeType.uint_8_8, PixelFormats.Uint_8_8 },
            { texture_info_quantize_typeType.snorm_8_8, PixelFormats.Snorm_8_8 },
            { texture_info_quantize_typeType.sint_8_8, PixelFormats.Sint_8_8 },
            { texture_info_quantize_typeType.unorm_5_6_5, PixelFormats.Unorm_5_6_5 },
            { texture_info_quantize_typeType.unorm_5_5_5_1, PixelFormats.Unorm_5_5_5_1 },
            { texture_info_quantize_typeType.unorm_4_4_4_4, PixelFormats.Unorm_4_4_4_4 },
            { texture_info_quantize_typeType.unorm_1_5_5_5, PixelFormats.Unorm_1_5_5_5 },
            { texture_info_quantize_typeType.uint_32, PixelFormats.Uint_32 },
            { texture_info_quantize_typeType.sint_32, PixelFormats.Sint_32 },
            { texture_info_quantize_typeType.float_32, PixelFormats.Float_32 },
            { texture_info_quantize_typeType.unorm_16_16, PixelFormats.Unorm_16_16 },
            { texture_info_quantize_typeType.uint_16_16, PixelFormats.Uint_16_16 },
            { texture_info_quantize_typeType.snorm_16_16, PixelFormats.Snorm_16_16 },
            { texture_info_quantize_typeType.sint_16_16, PixelFormats.Sint_16_16 },
            { texture_info_quantize_typeType.float_16_16, PixelFormats.Float_16_16 },
            { texture_info_quantize_typeType.float_11_11_10, PixelFormats.Float_11_11_10 },
            { texture_info_quantize_typeType.unorm_10_10_10_2, PixelFormats.Unorm_10_10_10_2 },
            { texture_info_quantize_typeType.uint_10_10_10_2, PixelFormats.Uint_10_10_10_2 },
            { texture_info_quantize_typeType.unorm_8_8_8_8, PixelFormats.Unorm_8_8_8_8 },
            { texture_info_quantize_typeType.uint_8_8_8_8, PixelFormats.Uint_8_8_8_8 },
            { texture_info_quantize_typeType.snorm_8_8_8_8, PixelFormats.Snorm_8_8_8_8 },
            { texture_info_quantize_typeType.sint_8_8_8_8, PixelFormats.Sint_8_8_8_8 },
            { texture_info_quantize_typeType.srgb_8_8_8_8, PixelFormats.Srgb_8_8_8_8 },
            { texture_info_quantize_typeType.unorm_2_10_10_10, PixelFormats.Unorm_2_10_10_10 },
            { texture_info_quantize_typeType.uint_2_10_10_10, PixelFormats.Uint_2_10_10_10 },
            { texture_info_quantize_typeType.uint_32_32, PixelFormats.Uint_32_32 },
            { texture_info_quantize_typeType.sint_32_32, PixelFormats.Sint_32_32 },
            { texture_info_quantize_typeType.float_32_32, PixelFormats.Float_32_32 },
            { texture_info_quantize_typeType.unorm_16_16_16_16, PixelFormats.Unorm_16_16_16_16 },
            { texture_info_quantize_typeType.uint_16_16_16_16, PixelFormats.Uint_16_16_16_16 },
            { texture_info_quantize_typeType.snorm_16_16_16_16, PixelFormats.Snorm_16_16_16_16 },
            { texture_info_quantize_typeType.sint_16_16_16_16, PixelFormats.Sint_16_16_16_16 },
            { texture_info_quantize_typeType.float_16_16_16_16, PixelFormats.Float_16_16_16_16 },
            { texture_info_quantize_typeType.uint_32_32_32_32, PixelFormats.Uint_32_32_32_32 },
            { texture_info_quantize_typeType.sint_32_32_32_32, PixelFormats.Sint_32_32_32_32 },
            { texture_info_quantize_typeType.float_32_32_32_32, PixelFormats.Float_32_32_32_32 },
            { texture_info_quantize_typeType.unorm_bc1, PixelFormats.Unorm_bc1 },
            { texture_info_quantize_typeType.srgb_bc1, PixelFormats.Srgb_bc1 },
            { texture_info_quantize_typeType.unorm_bc2, PixelFormats.Unorm_bc2 },
            { texture_info_quantize_typeType.srgb_bc2, PixelFormats.Srgb_bc2 },
            { texture_info_quantize_typeType.unorm_bc3, PixelFormats.Unorm_bc3 },
            { texture_info_quantize_typeType.srgb_bc3, PixelFormats.Srgb_bc3 },
            { texture_info_quantize_typeType.srgb_bc7, PixelFormats.Srgb_bc7 },
            { texture_info_quantize_typeType.unorm_bc4, PixelFormats.Unorm_bc4 },
            { texture_info_quantize_typeType.snorm_bc4, PixelFormats.Snorm_bc4 },
            { texture_info_quantize_typeType.unorm_bc5, PixelFormats.Unorm_bc5 },
            { texture_info_quantize_typeType.snorm_bc5, PixelFormats.Snorm_bc5 },
            { texture_info_quantize_typeType.ufloat_bc6, PixelFormats.Ufloat_bc6 },
            { texture_info_quantize_typeType.float_bc6, PixelFormats.Float_bc6 },
            { texture_info_quantize_typeType.unorm_bc7, PixelFormats.Unorm_bc7 },
            { texture_info_quantize_typeType.unorm_pvrtc1_2bpp, PixelFormats.Unorm_pvrtc1_2bpp },
            { texture_info_quantize_typeType.unorm_pvrtc1_4bpp, PixelFormats.Unorm_pvrtc1_4bpp },
            { texture_info_quantize_typeType.unorm_pvrtc1_alpha_2bpp, PixelFormats.Unorm_pvrtc1_alpha_2bpp },
            { texture_info_quantize_typeType.unorm_pvrtc1_alpha_4bpp, PixelFormats.Unorm_pvrtc1_alpha_4bpp },
            { texture_info_quantize_typeType.unorm_pvrtc2_alpha_2bpp, PixelFormats.Unorm_pvrtc2_alpha_2bpp },
            { texture_info_quantize_typeType.unorm_pvrtc2_alpha_4bpp, PixelFormats.Unorm_pvrtc2_alpha_4bpp },
            { texture_info_quantize_typeType.srgb_pvrtc1_2bpp, PixelFormats.Srgb_pvrtc1_2bpp },
            { texture_info_quantize_typeType.srgb_pvrtc1_4bpp, PixelFormats.Srgb_pvrtc1_4bpp },
            { texture_info_quantize_typeType.srgb_pvrtc1_alpha_2bpp, PixelFormats.Srgb_pvrtc1_alpha_2bpp },
            { texture_info_quantize_typeType.srgb_pvrtc1_alpha_4bpp, PixelFormats.Srgb_pvrtc1_alpha_4bpp },
            { texture_info_quantize_typeType.srgb_pvrtc2_alpha_2bpp, PixelFormats.Srgb_pvrtc2_alpha_2bpp },
            { texture_info_quantize_typeType.srgb_pvrtc2_alpha_4bpp, PixelFormats.Srgb_pvrtc2_alpha_4bpp },
            { texture_info_quantize_typeType.unorm_astc_4x4, PixelFormats.Unorm_astc_4x4 },
            { texture_info_quantize_typeType.srgb_astc_4x4, PixelFormats.Srgb_astc_4x4 },
            { texture_info_quantize_typeType.unorm_astc_5x4, PixelFormats.Unorm_astc_5x4 },
            { texture_info_quantize_typeType.srgb_astc_5x4, PixelFormats.Srgb_astc_5x4 },
            { texture_info_quantize_typeType.unorm_astc_5x5, PixelFormats.Unorm_astc_5x5 },
            { texture_info_quantize_typeType.srgb_astc_5x5, PixelFormats.Srgb_astc_5x5 },
            { texture_info_quantize_typeType.unorm_astc_6x5, PixelFormats.Unorm_astc_6x5 },
            { texture_info_quantize_typeType.srgb_astc_6x5, PixelFormats.Srgb_astc_6x5 },
            { texture_info_quantize_typeType.unorm_astc_6x6, PixelFormats.Unorm_astc_6x6 },
            { texture_info_quantize_typeType.srgb_astc_6x6, PixelFormats.Srgb_astc_6x6 },
            { texture_info_quantize_typeType.unorm_astc_8x5, PixelFormats.Unorm_astc_8x5 },
            { texture_info_quantize_typeType.srgb_astc_8x5, PixelFormats.Srgb_astc_8x5 },
            { texture_info_quantize_typeType.unorm_astc_8x6, PixelFormats.Unorm_astc_8x6 },
            { texture_info_quantize_typeType.srgb_astc_8x6, PixelFormats.Srgb_astc_8x6 },
            { texture_info_quantize_typeType.unorm_astc_8x8, PixelFormats.Unorm_astc_8x8 },
            { texture_info_quantize_typeType.srgb_astc_8x8, PixelFormats.Srgb_astc_8x8 },
            { texture_info_quantize_typeType.unorm_astc_10x5, PixelFormats.Unorm_astc_10x5 },
            { texture_info_quantize_typeType.srgb_astc_10x5, PixelFormats.Srgb_astc_10x5 },
            { texture_info_quantize_typeType.unorm_astc_10x6, PixelFormats.Unorm_astc_10x6 },
            { texture_info_quantize_typeType.srgb_astc_10x6, PixelFormats.Srgb_astc_10x6 },
            { texture_info_quantize_typeType.unorm_astc_10x8, PixelFormats.Unorm_astc_10x8 },
            { texture_info_quantize_typeType.srgb_astc_10x8, PixelFormats.Srgb_astc_10x8 },
            { texture_info_quantize_typeType.unorm_astc_10x10, PixelFormats.Unorm_astc_10x10 },
            { texture_info_quantize_typeType.srgb_astc_10x10, PixelFormats.Srgb_astc_10x10 },
            { texture_info_quantize_typeType.unorm_astc_12x10, PixelFormats.Unorm_astc_12x10 },
            { texture_info_quantize_typeType.srgb_astc_12x10, PixelFormats.Srgb_astc_12x10 },
            { texture_info_quantize_typeType.unorm_astc_12x12, PixelFormats.Unorm_astc_12x12 },
            { texture_info_quantize_typeType.srgb_astc_12x12, PixelFormats.Srgb_astc_12x12 },
        };

        /// <summary>
        /// 浮動小数点型テクスチャかテーブル
        /// </summary>
        private static readonly HashSet<texture_info_quantize_typeType> IsFloatTable = new HashSet<texture_info_quantize_typeType>
        {
            texture_info_quantize_typeType.float_16,
            texture_info_quantize_typeType.float_32,
            texture_info_quantize_typeType.float_16_16,
            texture_info_quantize_typeType.float_11_11_10,
            texture_info_quantize_typeType.float_32_32,
            texture_info_quantize_typeType.float_16_16_16_16,
            texture_info_quantize_typeType.float_32_32_32_32,
        };

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FtxTextureLoader()
        {
        }

        /// <summary>
        /// ファイルが読み込めるか調査する
        /// </summary>
        /// <param name="filePath">調査するファイルパス</param>
        /// <returns>読み込めるかどうか？</returns>
        public bool CanLoad(string filePath)
        {
            Debug.Assert(string.IsNullOrEmpty(filePath) == false, "パスの指定が不正");

            var ext = Path.GetExtension(filePath).ToLower();

            return (ext == ".ftxa") || (ext == ".ftxb");
        }

        /// <summary>
        /// テクスチャファイルをロードします。
        /// </summary>
        /// <param name="filePath">ロードするファイルパス</param>
        /// <returns>テクスチャのロード結果を返します。</returns>
        public LoadTextureResult Load(string filePath)
        {
            if (File.Exists(filePath) == false)
            {
                return new LoadTextureResult(LoadTextureResultCode.FileNotFound);
            }

            if (this.CanLoad(filePath) == false)
            {
                return new LoadTextureResult(LoadTextureResultCode.UnknowTextureType);
            }

            Version latestVersion = G3dIfUtility.GetLatestFileVersion();
            if (latestVersion == null)
            {
                return new LoadTextureResult(LoadTextureResultCode.UnknowTextureType);
            }

            // テクスチャファイルのバージョンをチェック
            if (G3dIfUtility.CheckG3DFileVersion(filePath, latestVersion) == false)
            {
                return new LoadTextureResult(LoadTextureResultCode.InvalidFileVersion);
            }

            LoadTextureResult result;

            // テクスチャファイルをロード
            try
            {
                result = this.LoadInternal(filePath);
            }
            catch (Exception e)
            {
                Logger.Log(LogLevels.Warning, "exception:{0}", e);

                return new LoadTextureResult(LoadTextureResultCode.FailedLoadingTexture);
            }

            return result;
        }

        /// <summary>
        /// 内部用のテクスチャロード処理を行います。
        /// </summary>
        /// <param name="filePath">ロードするファイルパス</param>
        /// <returns>テクスチャのロード結果を返します。</returns>
        private LoadTextureResult LoadInternal(string filePath)
        {
            var loadResult = G3dIfUtility.LoadG3dIf(filePath);
            var texture = loadResult.G3dIf.Item as textureType;
            Debug.Assert(texture != null, "テクスチャではない何かが読み込まれた。");

            G3dTextureData[][] g3dTextureData;

            if (texture.texture_info.depth == 1)
            {
                G3dTextureData[] textureData1d = TexUtilsProxy.ConvertTo1d2dStream(texture, loadResult.BinaryStreams);

                g3dTextureData = new G3dTextureData[1][] { textureData1d };
            }
            else
            {
                g3dTextureData = TexUtilsProxy.ConvertTo2dArrayStream(texture, loadResult.BinaryStreams);
            }

            Debug.Assert(TextureTypeConvertTable.ContainsKey(g3dTextureData[0][0].DimensionType), "型変換テーブルが不正。要調査");
            Debug.Assert(PixelFormatConvertTable.ContainsKey(g3dTextureData[0][0].QuantizeType), "型変換テーブルが不正。要調査");

            string comment = string.Empty;
            string color = string.Empty;
            if (loadResult.G3dIf.RootElement.comment != null)
            {
                comment = loadResult.G3dIf.RootElement.comment.text;
                color = loadResult.G3dIf.RootElement.comment.color;
            }

            var textureData = new FtxTextureData
            {
                // 以下、TextureData
                FilePath    = filePath,
                ModifyTime  = File.GetLastWriteTime(filePath),
                TextureType = TextureTypeConvertTable[g3dTextureData[0][0].DimensionType],
                PixelFormat = PixelFormatConvertTable[g3dTextureData[0][0].QuantizeType],
                IsFloat     = IsFloatTable.Contains(g3dTextureData[0][0].QuantizeType),
                Width       = g3dTextureData[0][0].Width,
                Height      = g3dTextureData[0][0].Height,
                Depth       = g3dTextureData[0][0].Depth,
                ArraySize   = texture.texture_info.depth,
                MipmapCount = texture.texture_info.mip_level,

                Comment     = comment,
                LabelColor  = color,

                // 以下、FtxTextureData
                DccPreset         = texture.texture_info.dcc_preset,
                Hint              = texture.texture_info.hint,
                Linear            = (bool[])texture.texture_info.linear.Clone(),
                WeightedCompress  = texture.texture_info.weighted_compress,
                InitialSwizzle    = texture.texture_info.initial_swizzle,
                MipGenFilter      = texture.texture_info.mip_gen_filter,
                CompSel           = this.ConvertComponentSelector(texture.texture_info.comp_sel),
                MipmapOffset      = texture.texture_info.level_offset,
                NativeImageStream = loadResult.BinaryStreams[texture.texture_info.stream_index].ByteData.ToArray(),
                ConverterName     = TexUtilsProxy.GetPlatformName(texture),
            };

            textureData.InitializeArray();

            var isFloat = g3dTextureData[0][0].IsFloat;

            if (g3dTextureData[0][0].OriginalImage != null)
            {
                // オリジナルイメージでデータの詰め替えが必要かどうか判定する
                bool needAtoG = false;

                // GenericConverterから得られるイメージは詰め替え不要なのでコンバータ名で判定する
                if (TexUtilsProxy.GetPlatformName(texture) == "OriginalImage")
                {
                    switch (textureData.PixelFormat)
                    {
                        case PixelFormats.Unorm_4_4:
                        case PixelFormats.Unorm_8_8:
                        case PixelFormats.Unorm_bc5:
                            var orgImgFmt = texture.original_image_array.Items[0].format;
                            if (!textureData.Hint.Equals("normal", StringComparison.InvariantCultureIgnoreCase) &&
                                (orgImgFmt == original_image_formatType.rgba8 ||
                                orgImgFmt == original_image_formatType.rgba32f))
                            {
                                needAtoG = true;
                            }

                            break;
                    }
                }

                Parallel.For(0, textureData.MipmapCount, j =>
                {
                    Parallel.For(0, textureData.ArraySize, i =>
                    {
                        // とりあえずそのまま格納
                        textureData.OriginalImages[i, j] = g3dTextureData[i][j].OriginalImage;

                        // ただし、条件に合致する2chデータに関しては詰め替えを行う
                        if (needAtoG)
                        {
                            var img = textureData.OriginalImages[i, j];
                            var imgData = img.LockBits(
                                new Rectangle(0, 0, img.Width, img.Height),
                                ImageLockMode.ReadWrite,
                                PixelFormat.Format32bppArgb);

                            unsafe
                            {
                                var imgPointer = (byte*)imgData.Scan0.ToPointer();
                                Parallel.For(0, img.Height, y =>
                                {
                                    for (int x = 0; x < imgData.Stride; x += 4)
                                    {
                                        int dstPos = y * imgData.Stride + x;
                                        imgPointer[dstPos + 1] = imgPointer[dstPos + 3];
                                    }
                                });
                            }

                            img.UnlockBits(imgData);
                        }
                    });
                });
            }
            else
            {
                // デコード配列がある場合はそちらを詰め込み
                Parallel.For(0, textureData.MipmapCount, j =>
                {
                    Parallel.For(0, textureData.ArraySize, i =>
                    {
                        if (isFloat)
                        {
                            textureData.SetFloatImage(i, j, g3dTextureData[i][j].FloatImage);
                        }
                        else
                        {
                            textureData.SetByteImage(i, j, g3dTextureData[i][j].ByteImage);
                        }
                    });
                });
            }

            // トリミング領域を計算
            textureData.TrimRect = this.CalcTrimRect(textureData);

            // タグの設定
            if (g3dTextureData[0][0].OriginalImage != null)
            {
                // オリジナルイメージが得られる場合はオリジナルイメージとして扱う
                // (一応PVRフォーマットである場合も考慮する)
                textureData.TextureTypeTag = textureData.PixelFormat.IsPvr() ? "PVRB" : "ORGB";
            }
            else
            {
                // オリジナルイメージが得られない=Cafeデコーダでデコードされているので、Cafeネイティブフォーマットとみなす
                // (一応PVRフォーマットである場合も考慮する)
                textureData.TextureTypeTag = textureData.PixelFormat.IsPvr() ? "PVRB" : "GX2B";
            }

            LoadTextureResult result = new LoadTextureResult()
            {
                ResultCode = LoadTextureResultCode.Success,
                TextureData = textureData
            };

            return result;
        }

        /// <summary>
        /// トリミング領域を計算します。
        /// </summary>
        /// <param name="textureData">テクスチャ</param>
        /// <returns>トリミング領域を返します。</returns>
        Rectangle CalcTrimRect(FtxTextureData textureData)
        {
            if (textureData.HasAlpha == false)
            {
                return new Rectangle(0, 0, textureData.Width, textureData.Height);
            }

            int[] trimPos = new int[] { int.MaxValue, int.MinValue, int.MaxValue, int.MinValue };

            for (int i = 0; i < textureData.ArraySize; ++i)
            {
                Bitmap srcImage = textureData.GenerateBitmap(i, 0);

                // アルファ値をRGB値に変換
                // TextureImageView.UpdateImages()を参考に実装
                ColorMatrix alphaMatrix = RenderUtility.CreateAlphaColorMatrix(textureData.CompSel, textureData.PixelFormat);
                Bitmap alphaImage = new Bitmap(srcImage.Width, srcImage.Height, PixelFormat.Format32bppArgb);

                using (var ia = new ImageAttributes())
                using (var g = Graphics.FromImage(alphaImage))
                {
                    ia.SetColorMatrix(alphaMatrix);

                    g.DrawImage(
                        srcImage,
                        new Rectangle(0, 0, srcImage.Width, srcImage.Height),
                        0,
                        0,
                        srcImage.Width,
                        srcImage.Height,
                        GraphicsUnit.Pixel,
                        ia);
                }

                // トリミング位置(Xmin, Xmax, Ymin, Ymax)を計算
                // TextureData.GenerateBitmapを参考にロック処理を実装
                Rectangle rect = new Rectangle(0, 0, srcImage.Width, srcImage.Height);
                BitmapData alphaData = alphaImage.LockBits(rect, ImageLockMode.ReadOnly, PixelFormat.Format32bppArgb);

                try
                {
                    unsafe
                    {
                        byte* alphaDataPtr = (byte*)alphaData.Scan0.ToPointer();

                        for (int y = 0; y < rect.Height; ++y)
                        {
                            for (int x = 0; x < rect.Width; ++x)
                            {
                                int index = (y * rect.Width * 4) + (x * 4);

                                // R要素だけチェック
                                // StyleCop
                                if (alphaDataPtr[index] == 0)
                                {
                                    continue;
                                }

                                // StyleCop
                                if (x < trimPos[0])
                                {
                                    trimPos[0] = x;
                                }

                                // StyleCop
                                if (x > trimPos[1])
                                {
                                    trimPos[1] = x;
                                }

                                // StyleCop
                                if (y < trimPos[2])
                                {
                                    trimPos[2] = y;
                                }

                                // StyleCop
                                if (y > trimPos[3])
                                {
                                    trimPos[3] = y;
                                }
                            }
                        }

                        if (trimPos[0] > trimPos[1] || trimPos[2] > trimPos[3])
                        {
                            trimPos[0] = trimPos[1] = trimPos[2] = trimPos[3] = 0;
                        }
                    }
                }
                finally
                {
                    alphaImage.UnlockBits(alphaData);
                }
            }

            // トリミング位置 → トリミング領域に変換
            return new Rectangle(trimPos[0], trimPos[2], trimPos[1] - trimPos[0] + 1, trimPos[3] - trimPos[2] + 1);
        }

        /// <summary>
        /// Convert component selector.
        /// </summary>
        /// <param name="compSel">The component selector.</param>
        /// <returns>The converted component selector.</returns>
        private ColorComponents[] ConvertComponentSelector(texture_info_comp_selValue[] compSel)
        {
            if (compSel == null)
            {
                return null;
            }

            var result = new ColorComponents[compSel.Length];
            for (int i = 0; i < compSel.Length; ++i)
            {
                result[i] = ComponentSelectorConvertTable[compSel[i]];
            }

            return result;
        }
    }
}
