﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Log;
using EffectMaker.ShaderConverterWrapper;

using LogLevelMapType = System.Collections.Generic.Dictionary<string, EffectMaker.Foundation.Log.LogLevels>;
using MessageHandlerMapType = System.Collections.Generic.Dictionary<string, System.Action<EffectMaker.ShaderConverterWrapper.DotNetMessage>>;

namespace EffectMaker.ShaderManager
{
    /// <summary>
    /// Message forwarder class that forwards messages send from
    /// the shader converter to the application for displaying or
    /// further process.
    /// </summary>
    public class ShaderConverterMessageForwarder : IDotNetMessageForwarder
    {
        /// <summary>A mapping table for converting log levels.</summary>
        private static readonly LogLevelMapType LogLevelMap = new LogLevelMapType()
        {
            { "debug", LogLevels.Debug },
            { "information", LogLevels.Information },
            { "warning", LogLevels.Warning },
            { "error", LogLevels.Error },
            { "fatal", LogLevels.Fatal },
        };

        /// <summary>
        /// A dictionary that maps messages names with their handler.
        /// This dictionary maps only special messages that needs to be handled separately.
        /// </summary>
        private static readonly MessageHandlerMapType SpecialMessageHandlers = new MessageHandlerMapType()
        {
            { "ShaderConverterMsg_OpenTextFile",   OpenTextFile },
            { "ShaderConverterMsg_ProfilerTime",   OutputProfilerTime },
            { "ShaderMsg_ShaderExceedsBufferSize", OutputShaderExceedsBufferSizeMessage },
            { "ShaderMsg_General",                 OutputStrings },
        };

        /// <summary>
        /// Forward message sent from the shader converter library.
        /// </summary>
        /// <param name="msgData">The message data to output.</param>
        public void ForwardMessage(DotNetMessage msgData)
        {
            Action<DotNetMessage> handler;
            if (SpecialMessageHandlers.TryGetValue(msgData.Name, out handler) == true)
            {
                handler(msgData);
            }
            else
            {
                OutputMessageFromStringTable(msgData);
            }
        }

        /// <summary>
        /// Open text file with the text editor that is assigned in the config.
        /// </summary>
        /// <param name="msgData">The message data that contains the file path of the text file.</param>
        private static void OpenTextFile(DotNetMessage msgData)
        {
            ApplicationIOManager.OpenTextFileWithExternalTextEditor(msgData.GetArg("FilePath"));
        }

        /// <summary>
        /// Open text file with the text editor that is assigned in the config.
        /// </summary>
        /// <param name="msgData">The message data that contains the file path of the text file.</param>
        private static void OutputProfilerTime(DotNetMessage msgData)
        {
            // Output the message.
            Logger.Log(
                "DebugConsole,Console",
                LogLevels.Profile,
                "(shader converter profiler) {1} seconds spent for '{0}'.",
                msgData.GetArg("Tag"),
                msgData.GetArg("ElapsedTime"));
        }

        /// <summary>
        /// Output message.
        /// 1. Get the format string from string resource with the message name.
        /// 2. Compose the message with the string get from the resource and
        ///    the arguments (Destination and LogLevel are not included) in the message data.
        /// 3. Log out the message with Destination and LogLevel argument in the message data.
        /// </summary>
        /// <param name="msgData">The message data.</param>
        private static void OutputMessageFromStringTable(DotNetMessage msgData)
        {
            string msg = Properties.Resources.ResourceManager.GetString(msgData.Name);
            if (string.IsNullOrEmpty(msg) == true)
            {
                return;
            }

            if (msgData.Arguments.Length > 2)
            {
                // Copy the arguments except the first two.
                int copyLength = msgData.Arguments.Length - 2;
                string[] arguments = new string[copyLength];
                Array.Copy(msgData.Arguments, 2, arguments, 0, copyLength);

                // Compose the message.
                msg = string.Format(msg, (object[])arguments);
            }

            // Get destination and log level from the message data.
            string dest = msgData.GetArg("Destination");
            string level = msgData.GetArg("LogLevel");
            if (dest == null || level == null)
            {
                return;
            }

            // Convert the log level from string.
            LogLevels logLevel;
            if (LogLevelMap.TryGetValue(level.ToLower(), out logLevel) == false)
            {
                return;
            }

            // Output the message.
            Logger.Log(dest, logLevel, msg);
        }

        /// <summary>
        /// Output error message when converted shader exceeds the buffer size.
        /// </summary>
        /// <param name="msgData">The message data.</param>
        private static void OutputShaderExceedsBufferSizeMessage(DotNetMessage msgData)
        {
            // Output the message.
            Logger.Log(
                LogLevels.Error,
                Properties.Resources.ShaderMsg_ShaderExceedsBufferSize,
                msgData.GetArg("ShaderSize"));
        }

        /// <summary>
        /// Output user generated message.
        /// </summary>
        /// <param name="msgData">The message data that contains the file path of the text file.</param>
        private static void OutputStrings(DotNetMessage msgData)
        {
            // Get destination and log level from the message data.
            string dest = msgData.GetArg("Destination");
            string level = msgData.GetArg("LogLevel");
            if (dest == null || level == null)
            {
                return;
            }

            // Convert the log level from string.
            LogLevels logLevel;
            if (LogLevelMap.TryGetValue(level.ToLower(), out logLevel) == false)
            {
                return;
            }

            // Output the message.
            Logger.Log(dest, logLevel, msgData.GetArg("OutputMsg"));
        }
    }
}
