﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryData;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.Utilities
{
    /// <summary>
    /// Class that processes modified binary fields, collect needed information and send
    /// modification to the viewer.
    /// </summary>
    public class SendModifiedBinaryContext
    {
        /// <summary>The count of begun sessions.</summary>
        private static int beginSessionCount = 0;

        /// <summary>
        /// The dictionary that maps root binary data of the modified fields and contexts.
        /// </summary>
        private static Dictionary<BinaryStructInstance, SendModifiedBinaryContext> contextMap =
            new Dictionary<BinaryStructInstance, SendModifiedBinaryContext>();

        /// <summary>
        /// The flag indicating whether to send the whole binary data or
        /// just the modified part to the viewer.
        /// </summary>
        private bool sendFullBinary = false;

        /// <summary>The root binary data structure of the modified fields.</summary>
        private BinaryStructInstance rootBinaryStruct;

        /// <summary>A queue that holds all the modified binary fields.</summary>
        private Queue<BinaryFieldInstance> modifiedFieldQueue = new Queue<BinaryFieldInstance>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="rootBinaryStruct">The root binary data structure of the modified fields.</param>
        private SendModifiedBinaryContext(BinaryStructInstance rootBinaryStruct)
        {
            this.rootBinaryStruct = rootBinaryStruct;
        }

        /// <summary>
        /// バイナリリロードが発生した時にトリガーされるイベントです。
        /// </summary>
        public static event EventHandler SendBinaryRequested;

        /// <summary>
        /// Begin a session that process modified binary fields and send all the modified
        /// data to the viewer when the sessions are finished.
        /// </summary>
        public static void BeginSession()
        {
            if (beginSessionCount <= 0)
            {
                beginSessionCount = 0;
                contextMap.Clear();
            }

            ++beginSessionCount;
        }

        /// <summary>
        /// Add a binary structure to send to the viewer.
        /// </summary>
        /// <param name="rootBinaryStruct">The binary structure instance.</param>
        public static void AddBinaryStructToSend(BinaryStructInstance rootBinaryStruct)
        {
            if (rootBinaryStruct == null)
            {
                return;
            }

            SendModifiedBinaryContext context = null;

            // There is no begun session, just send the field data to the viewer.
            if (beginSessionCount <= 0)
            {
                context = new SendModifiedBinaryContext(rootBinaryStruct);
                context.sendFullBinary = true;
                context.SendModificationToViewer();

                return;
            }

            // Find the context for the root binary data structure.
            if (contextMap.TryGetValue(rootBinaryStruct, out context) == false)
            {
                context = new SendModifiedBinaryContext(rootBinaryStruct);
                contextMap.Add(rootBinaryStruct, context);
            }

            context.sendFullBinary = true;
        }

        /// <summary>
        /// Add a modified binary field, the messages will be sent to the viewer
        /// when the sessions are finished.
        /// </summary>
        /// <param name="field">The modified binary field instance.</param>
        public static void AddModifiedField(BinaryFieldInstance field)
        {
            var rootBinaryStruct = field.FindRootElement() as BinaryStructInstance;
            if (rootBinaryStruct == null)
            {
                return;
            }

            SendModifiedBinaryContext context = null;

            // There is no begun session, just send the field data to the viewer.
            if (beginSessionCount <= 0)
            {
                context = new SendModifiedBinaryContext(rootBinaryStruct);
                context.EnqueueModifiedBinaryField(field);
                context.SendModificationToViewer();

                return;
            }

            // Find the context for the root binary data structure.
            if (contextMap.TryGetValue(rootBinaryStruct, out context) == false)
            {
                context = new SendModifiedBinaryContext(rootBinaryStruct);
                contextMap.Add(rootBinaryStruct, context);
            }

            // Enqueue the modified binary field to the context, wait until everything is
            // updated and all the information is collected, then send all the data together.
            context.EnqueueModifiedBinaryField(field);
        }

        /// <summary>
        /// Finish a session.
        /// When all the sessions are finished, the messages will be sent to the viewer
        /// to upload the modified binary data.
        /// </summary>
        public static void EndSession()
        {
            if (beginSessionCount <= 0)
            {
                beginSessionCount = 0;
                return;
            }

            --beginSessionCount;
            if (beginSessionCount == 0)
            {
                // All sessions are finished, send the modified binary data.
                foreach (SendModifiedBinaryContext context in contextMap.Values)
                {
                    context.SendModificationToViewer();
                }
            }
        }

        /// <summary>
        /// Enqueue the binary field instance that is modified, so that we
        /// can collect all the information we need, and send the messages
        /// to the viewer.
        /// </summary>
        /// <param name="field">The modified binary field.</param>
        private void EnqueueModifiedBinaryField(BinaryFieldInstance field)
        {
            // If the context already needs to send the full binary data,
            // it's pointless to enqueue other fields anymore.
            if (this.sendFullBinary == true)
            {
                return;
            }

            if (field.Definition.SendModificationType == SendModificationTypes.FullBinary)
            {
                this.sendFullBinary = true;
                this.modifiedFieldQueue.Clear();
            }

            this.modifiedFieldQueue.Enqueue(field);
        }

        /// <summary>
        /// All the needed information is gathered, send the modified data to the viewer.
        /// </summary>
        /// <returns>True on success.</returns>
        private bool SendModificationToViewer()
        {
            if (this.rootBinaryStruct.DataModel == null)
            {
                return false;
            }

            if (this.sendFullBinary == true)
            {
                if (this.rootBinaryStruct.DataModel is EmitterSetData)
                {
                    if (SendBinaryRequested != null)
                    {
                        SendBinaryRequested(this, EventArgs.Empty);
                    }

                    return ViewerMessageHelper.SendEmitterSet(
                        (EmitterSetData)this.rootBinaryStruct.DataModel);
                }
                else
                {
                    // Send binary does not support this kind of data model yet.
                    return false;
                }
            }
            else
            {
                bool finalResult = true;
                while (this.modifiedFieldQueue.Count > 0)
                {
                    // Get the modified binary field.
                    BinaryFieldInstance field = this.modifiedFieldQueue.Dequeue();

                    // Send the modified data to the viewer.
                    bool result = ViewerMessageHelper.SendModifiedData(this.rootBinaryStruct, field);

                    // Is the message sent successfully?
                    if (result == false)
                    {
                        finalResult = false;
                        Logger.Log(
                            LogLevels.Warning,
                            "SendModifiedBinaryContext.SendModificationToViewer : Failed sending modified data of {0} to viewer.",
                            this.rootBinaryStruct.DataModel.GetType().Name);
                    }
                }

                return finalResult;
            }
        }
    }
}
