﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.DataModelLogic.BinaryConversionInfo;
using EffectMaker.DataModelLogic.BinaryConverters;
using EffectMaker.DataModelLogic.Utilities;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Binary data field that selects underlying binary structure definition according to
    /// the input data type.
    /// </summary>
    public class SelectBinaryDataFieldInstance : BinaryFieldInstance
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">The parent binary data structure instance.</param>
        /// <param name="definition">The binary data field definition.</param>
        public SelectBinaryDataFieldInstance(
            BinaryStructInstance parent,
            BinaryFieldDefinition definition) :
            base(parent, definition)
        {
        }

        /// <summary>
        /// Convert binary data from the data model.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool ConvertBinary()
        {
            this.IsConvertingBinary = true;

            this.CheckFieldOffsetForDebugBreak();

            try
            {
                if ((this.Definition.ConverterDefinition == null) ||
                    (this.Definition.InputProperties == null) ||
                    (this.Definition.InputProperties.Length <= 0))
                {
                    return false;
                }

                // Create the converter.
                var converter = this.Definition.ConverterDefinition.CreateConverter();

                // The required input value count of the converter
                // should match the input property count.
                int inputValueCount = this.Definition.InputProperties.Length;
                if (converter.MinInputValueCount > inputValueCount ||
                    converter.MaxInputValueCount < inputValueCount)
                {
                    Logger.Log(LogLevels.Warning, "SelectBinaryDataFieldInstance.ConvertBinary : Input value count does not match the requested input count of the converter.");
                    return false;
                }

                // Feed input values to the converter.
                if (this.FeedInputValueToConverter(converter) == false)
                {
                    return false;
                }

                int originalBinarySize = this.Size;

                // Clear the binary structures first.
                this.ClearUnderlyingBinaryStructures();

                // Convert!!!
                if (converter.Convert() == false)
                {
                    Logger.Log(LogLevels.Warning, "SelectBinaryDataFieldInstance.ConvertBinary : The converter reported a failure when converting the data.");
                    return false;
                }

                object output = converter.GetOutputValue();

                if (output == null)
                {
                    // There is nothing to write, just skip this field.
                    this.Size = 0;
                    return true;
                }
                else if ((output is IEnumerable<DataModelBase>) == false &&
                         (output is DataModelBase) == false)
                {
                    Logger.Log(LogLevels.Warning, "SelectBinaryDataFieldInstance.ConvertBinary : The output binary field is a binary data, however the assigned converter does not output binary data.");
                    return false;
                }

                // Convert the single data model to an IEnumerable
                // so we don't have to write redundant code to process
                // the single item case.
                if (output is DataModelBase)
                {
                    output = new DataModelBase[] { (DataModelBase)output };
                }

                bool result = false;
                if (output is IEnumerable<DataModelBase>)
                {
                    var dataModels = output as IEnumerable<DataModelBase>;

                    // Convert all the binary data structures in the array.
                    int offset = 0;
                    int offsetIncludeHeader = 0;
                    foreach (DataModelBase dataModel in dataModels)
                    {
                        var binaryStructDef =
                            BinaryStructDefinition.FindBinaryStructureDefinition(dataModel.GetType());
                        if (binaryStructDef == null)
                        {
                            Logger.Log(LogLevels.Warning, "SelectBinaryDataFieldInstance.ConvertBinary : The input data model of type {0} has no binary structure that takes it as source data model.", dataModel.GetType().Name);
                            return false;
                        }

                        // Create the binary structure.
                        var element = binaryStructDef.CreateInstance(this, dataModel);

                        this.AddBinaryStructureInstance(element);

                        // Setup the binary data offset of the binary structure.
                        element.LocalOffset = offsetIncludeHeader;
                        element.Offset = offset + this.Offset;

                        // Convert the element.
                        if (element.ConvertBinary() == false)
                        {
                            return false;
                        }

                        // Increment.
                        offset += element.Size;
                        offsetIncludeHeader += element.SizeIncludeHeader;
                    }

                    // Update the binary size.
                    this.Size = offset;
                    this.SizeIncludeHeader = offsetIncludeHeader;

                    if (this.Parent != null && offset != originalBinarySize)
                    {
                        // Report size change.
                        this.Parent.ReportSizeChanged(this);
                    }

                    result = true;
                }

                return result;
            }
            finally
            {
                this.IsConvertingBinary = false;
            }
        }

        /// <summary>
        /// Write the binary data for the given session.
        /// </summary>
        /// <param name="session">The WriteBinaryDataSession that is being processed now.</param>
        /// <returns>True on success.</returns>
        public override bool WriteBinaryData(WriteBinaryDataSession session)
        {
            if (this.UnderlyingBinaryStructures != null)
            {
                using (new WriteBinaryDataBlock(session, this))
                {
                    foreach (BinaryStructInstance element in this.UnderlyingBinaryStructures)
                    {
                        if (element.WriteBinaryData(session) == false)
                        {
                            return false;
                        }
                    }
                }
            }

            return true;
        }
    }
}
