﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.Foundation.Collections;

namespace EffectMaker.DataModelLogic.BinaryData
{
    /// <summary>
    /// Class for the definition of a binary data field.
    /// </summary>
    public class BinaryFieldDefinition
    {
        /// <summary>
        /// Static constructor.
        /// </summary>
        static BinaryFieldDefinition()
        {
            EnableBinaryFieldDebug = false;
            BreakOnFieldOffset = -1;
            BreakOnInputPropertyGuid = Guid.Empty;
            ////BreakOnInputPropertyGuid = new Guid("");
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="sendModificationType">The type of message to send if data is modified.</param>
        /// <param name="converterDef">The converter definition.</param>
        /// <param name="structDef">The definition of a binary structure for the field.</param>
        /// <param name="inputProperties">The array of definition of the input properties.</param>
        public BinaryFieldDefinition(
            SendModificationTypes sendModificationType,
            ConverterDefinition converterDef,
            BinaryStructDefinition structDef,
            params InputPropertyDefinition[] inputProperties)
        {
            this.SendModificationType = sendModificationType;
            this.ConverterDefinition = converterDef;
            this.StructDefinition = structDef;
            this.InputProperties = inputProperties;
        }

        /// <summary>
        /// Get or set the flag indicating whether to enable binary field debugging.
        /// </summary>
        public static bool EnableBinaryFieldDebug { get; set; }

        /// <summary>
        /// Get or set the binary field offset, where the execution breaks if
        /// the field starts from the offset.
        /// </summary>
        public static int BreakOnFieldOffset { get; set; }

        /// <summary>
        /// Get or set the the data model property Guid, where if not Guid.Empty,
        /// the execution breaks if the field uses this property as input.
        /// </summary>
        public static Guid BreakOnInputPropertyGuid { get; set; }

        /// <summary>
        /// Get or set the type of message to send if the data in the binary field is modified.
        /// </summary>
        public SendModificationTypes SendModificationType { get; set; }

        /// <summary>
        /// Get or set the tag of the binary field.
        /// </summary>
        public string Tag { get; set; }

        /// <summary>
        /// Get or set the converter definition.
        /// </summary>
        public ConverterDefinition ConverterDefinition { get; set; }

        /// <summary>
        /// Get or set the struct definition of this binary field.
        /// </summary>
        public BinaryStructDefinition StructDefinition { get; set; }

        /// <summary>
        /// Get or set the array of the definition of the input data model properties.
        /// </summary>
        public InputPropertyDefinition[] InputProperties { get; set; }

        /// <summary>
        /// Create an instance with the definition.
        /// </summary>
        /// <param name="parent">The parent binary data structure instance.</param>
        /// <returns>The created instance.</returns>
        public virtual BinaryFieldInstance CreateInstance(BinaryStructInstance parent)
        {
            return new BinaryFieldInstance(parent, this);
        }
    }
}
