﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// 文字列をコンバートします。
    /// </summary>
    public class StringConverter : ConverterBase
    {
        /// <summary>
        /// コンバートに使用する文字エンコーディングです。
        /// </summary>
        public static readonly System.Text.Encoding Encoding;

        /// <summary>
        /// staticコンストラクタです。
        /// </summary>
        static StringConverter()
        {
            Encoding = System.Text.Encoding.GetEncoding("Shift_JIS");
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public StringConverter() :
            base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.StringConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.StringConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.StringConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            if (this.IsInputValueAssigned(0) == false)
            {
                Logger.Log(LogLevels.Warning, "StringConverter.Convert : There is no input value assigned.");
                return false;
            }

            var str = this.GetInputValue(0) as string;
            if (str == null)
            {
                Logger.Log(LogLevels.Warning, "StringConverter.Convert : The assigned input value is not a string.");
                return false;
            }

            // 文字列をバイト配列に変換
            var output = Encoding.GetBytes(str);

            this.SetOutputValue(output);

            return true;
        }
    }
}
