﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Class that holds the parameters of the binary converters.
    /// </summary>
    public class ConverterParamHolder
    {
        /// <summary>The parameters for the converter.</summary>
        private Dictionary<string, ParameterData> parameters =
            new Dictionary<string, ParameterData>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public ConverterParamHolder()
        {
        }

        /// <summary>
        /// Register parameter for the converter.
        /// </summary>
        /// <param name="name">The name of the parameter.</param>
        /// <param name="description">The description for the parameter.</param>
        /// <param name="validator">The value validator for the parameter.</param>
        public void RegisterParameter(
            string name,
            string description,
            Func<object, bool> validator)
        {
            // Is this parameter already added?
            if (this.parameters.ContainsKey(name) == true)
            {
                return;
            }

            // Add the parameter data.
            this.parameters.Add(
                name,
                new ParameterData(description, validator));
        }

        /// <summary>
        /// Enumerate the names of the parameters accepted by the converter.
        /// </summary>
        /// <returns>The parameter names.</returns>
        public IEnumerable<string> EnumerateParameterNames()
        {
            return this.parameters.Select(pair => pair.Key);
        }

        /// <summary>
        /// Enumerate the name and description of the parameters accepted by the converter.
        /// </summary>
        /// <returns>The name and description of the parameters.</returns>
        public IEnumerable<KeyValuePair<string, string>> EnumerateParameterNameAndDesc()
        {
            return this.parameters.Select(pair =>
                new KeyValuePair<string, string>(pair.Key, pair.Value.Description));
        }

        /// <summary>
        /// Check if the parameter is valid.
        /// </summary>
        /// <param name="name">The name of the parameter.</param>
        /// <param name="parameter">The value of the parameter.</param>
        /// <returns>False if the given parameter value is invalid, or the parameter name does not exist.</returns>
        public bool ValidateParameter(string name, object parameter)
        {
            // Get the parameter data from our dictionary.
            ParameterData data;
            if (this.parameters.TryGetValue(name, out data) == false)
            {
                return false;
            }

            // Validate the parameter value.
            if (data.Validator != null && data.Validator(parameter) == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Set parameter for the converter.
        /// </summary>
        /// <param name="name">The name of the parameter.</param>
        /// <param name="parameter">The value of the parameter.</param>
        /// <returns>True on success.</returns>
        public bool SetParameter(string name, object parameter)
        {
            // Get the parameter data from our dictionary.
            ParameterData data;
            if (this.parameters.TryGetValue(name, out data) == false)
            {
                return false;
            }

            // Validate the parameter value.
            if (data.Validator != null && data.Validator(parameter) == false)
            {
                return false;
            }

            // Set parameter value.
            this.parameters[name].Value = parameter;

            return true;
        }

        /// <summary>
        /// Get the parameter value of the designated name.
        /// </summary>
        /// <param name="name">The parameter name.</param>
        /// <param name="value">The parameter value.</param>
        /// <returns>False if the parameter is not found or set.</returns>
        public bool GetParameter(string name, out object value)
        {
            value = null;
            if (this.parameters.ContainsKey(name) == false)
            {
                return false;
            }

            ParameterData data = this.parameters[name];
            if (data == null || data.IsValueSet == false)
            {
                return false;
            }

            value = this.parameters[name].Value;

            return true;
        }

        /// <summary>
        /// Unset all the parameter values.
        /// </summary>
        public void UnsetParameters()
        {
            foreach (ParameterData data in this.parameters.Select(pair => pair.Value))
            {
                data.UnsetValue();
            }
        }

        /// <summary>
        /// Class that holds the parameter data.
        /// </summary>
        private class ParameterData
        {
            /// <summary>The parameter value.</summary>
            private object value = null;

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="description">The description of the parameter.</param>
            /// <param name="validator">The value validator for the parameter.</param>
            public ParameterData(string description, Func<object, bool> validator)
            {
                this.Description = description;
                this.Validator = validator;
                this.IsValueSet = false;
            }

            /// <summary>
            /// Get the description of the parameter.
            /// </summary>
            public string Description { get; private set; }

            /// <summary>
            /// Get the value validator for the parameter.
            /// </summary>
            public Func<object, bool> Validator { get; private set; }

            /// <summary>
            /// Get the flag indicating whether the parameter value is set or not.
            /// </summary>
            public bool IsValueSet { get; private set; }

            /// <summary>
            /// Get or set the parameter value.
            /// </summary>
            public object Value
            {
                get
                {
                    return this.value;
                }

                set
                {
                    this.IsValueSet = true;
                    this.value = value;
                }
            }

            /// <summary>
            /// Unset the parameter value.
            /// The IsValueSet flag will be reset to false,
            /// and the value will be set to null.
            /// </summary>
            public void UnsetValue()
            {
                this.value = null;
                this.IsValueSet = false;
            }
        }
    }
}
