﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using ComboBoxItemType = System.Collections.Generic.KeyValuePair<string, object>;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// コンボボックスアイテム内から指定された文字列を固定長配列にコンバートするクラスです。
    /// </summary>
    public class ComboBoxStringConverter : ConverterBase
    {
        /// <summary>Constant for char array length parameter name.</summary>
        private const string CharArrayLengthParamName = "CharArrayLength";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static ComboBoxStringConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                CharArrayLengthParamName,
                "The output element count.",
                ValidateCharArrayLengthParamter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public ComboBoxStringConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 2; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.StringComboBoxConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.StringComboBoxConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.StringComboBoxConverterInputValue0Desc;
            }
            else if (index == 1)
            {
                return Properties.Resources.StringComboBoxConverterInputValue1Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            var items = this.GetInputValue(1) as IEnumerable<ComboBoxItemType>;
            int index = -1;
            object indexInput = this.GetInputValue(0);
            if (indexInput is string)
            {
                int.TryParse((string)indexInput, out index);
            }
            else if (indexInput is int)
            {
                index = (int)indexInput;
            }

            if (items == null || index == -1)
            {
                return false;
            }

            string input = items.First(i => (int)i.Value == index).Key;

            object param;

            // Get the char array length from the parameter.
            int charArrayLength = 32;
            if (this.GetParameter(CharArrayLengthParamName, out param) == true)
            {
                charArrayLength = (int)param;
            }

            // First initialize the array with zeros.
            var output = Enumerable.Repeat('\0', charArrayLength).ToArray();

            // Output value as a character array with a maximum length of 32 characters.
            input.ToCharArray(0, Math.Min(input.Length, charArrayLength - 1)).CopyTo(output, 0);

            this.SetOutputValue(output);

            return true;
        }

        /// <summary>
        /// Validate char array length parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateCharArrayLengthParamter(object value)
        {
            int count;
            if (value is int)
            {
                count = (int)value;
            }
            else if (value is string)
            {
                if (int.TryParse((string)value, out count) == false)
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            if (count <= 0)
            {
                return false;
            }

            return true;
        }
    }
}
