﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel.Attributes;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.DataModelLogic.BinaryData;
using EffectMaker.DataModelLogic.DataModelProxies;
using EffectMaker.DataModelLogic.Events;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelLogic.BinaryConversionInfo
{
    /// <summary>
    /// Manager class for the binary conversion informations.
    /// </summary>
    public class BinaryConversionInfoManager
    {
        /// <summary>
        /// The dictionary that maps data model property Guid and
        /// corresponding binary field instances.
        /// </summary>
        private static Dictionary<Guid, LinkedList<BinaryFieldInstance>> propertyFieldMap =
            new Dictionary<Guid, LinkedList<BinaryFieldInstance>>();

        /// <summary>
        /// The dictionary that maps binary struct instances with their source
        /// data model's Guid.
        /// </summary>
        private static Dictionary<Guid, BinaryStructInstance> binaryDataMap =
            new Dictionary<Guid, BinaryStructInstance>();

        /// <summary>
        /// Event triggered when a data model instance is created.
        /// </summary>
        public static event EventHandler<DataModelBase> DataModelCreated = null;

        /// <summary>
        /// Event triggered when a data model instance is destroyed.
        /// </summary>
        public static event EventHandler<DataModelBase> DataModelDestroyed = null;

        /// <summary>
        /// Report that a data model instance is created.
        /// </summary>
        /// <param name="dataModel">The created data model.</param>
        public static void ReportDataModelCreated(DataModelBase dataModel)
        {
            // Trigger the event.
            if (DataModelCreated != null)
            {
                DataModelCreated(null, dataModel);
            }
        }

        /// <summary>
        /// Report that a data model instance is destroyed.
        /// </summary>
        /// <param name="dataModel">The destroyed data model.</param>
        public static void ReportDataModelDestroyed(DataModelBase dataModel)
        {
            // Trigger the event.
            if (DataModelDestroyed != null)
            {
                DataModelDestroyed(null, dataModel);
            }
        }

        /// <summary>
        /// Find the binary data with the source data model.
        /// </summary>
        /// <param name="sourceDataModel">The source data model.</param>
        /// <returns>The binary data which the source data model is as specified.</returns>
        public static BinaryStructInstance FindBinaryData(DataModelBase sourceDataModel)
        {
            if (sourceDataModel == null)
            {
                return null;
            }

            BinaryStructInstance binaryData;
            if (binaryDataMap.TryGetValue(sourceDataModel.Guid, out binaryData) == false)
            {
                return null;
            }

            return binaryData;
        }

        /// <summary>
        /// Register binary data.
        /// </summary>
        /// <param name="binaryData">The binary data.</param>
        public static void RegisterBinaryData(BinaryStructInstance binaryData)
        {
            if (binaryData == null || binaryData.DataModel == null)
            {
                return;
            }

            if (binaryDataMap.ContainsKey(binaryData.DataModel.Guid) == true)
            {
                Logger.Log(LogLevels.Warning, "BinaryConversionInfoManager.RegisterBinaryData : Cannot register multiple binary data with the same source data model.");
                return;
            }

            binaryDataMap.Add(binaryData.DataModel.Guid, binaryData);
        }

        /// <summary>
        /// Unregister the binary data.
        /// </summary>
        /// <param name="binaryData">The binary data.</param>
        public static void UnregisterBinaryData(BinaryStructInstance binaryData)
        {
            if (binaryData == null || binaryData.DataModel == null)
            {
                return;
            }

            binaryDataMap.Remove(binaryData.DataModel.Guid);
        }

        /// <summary>
        /// Register the Guid of the data model property with a binary field instance
        /// of which the field instance needs to update when the data model property is
        /// modified.
        /// </summary>
        /// <param name="field">The binary field instance.</param>
        /// <param name="dataModelPropertyGuid">The Guid of the data model property.</param>
        public static void RegisterBinaryFieldInstance(
            BinaryFieldInstance field,
            Guid dataModelPropertyGuid)
        {
            LinkedList<BinaryFieldInstance> list;
            if (propertyFieldMap.TryGetValue(dataModelPropertyGuid, out list) == false)
            {
                // The data model property Guid has not been added, add it.
                list = new LinkedList<BinaryFieldInstance>();

                propertyFieldMap.Add(dataModelPropertyGuid, list);
            }

            // Add the binary field instance to the linked list.
            list.AddLast(field);
        }

        /// <summary>
        /// Unregister the binary field instance, so it doesn't handle the data model
        /// property change event anymore.
        /// </summary>
        /// <param name="field">The binary field instance.</param>
        /// <param name="dataModelPropertyGuid">The Guid of the data model property.</param>
        public static void UnregisterBinaryFieldInstance(
            BinaryFieldInstance field,
            Guid dataModelPropertyGuid)
        {
            LinkedList<BinaryFieldInstance> list;
            if (propertyFieldMap.TryGetValue(dataModelPropertyGuid, out list) == false)
            {
                // The binary field instance is not registered to the data model property.
                return;
            }

            // Remove the binary field instance from our list.
            list.Remove(field);
        }

        /// <summary>
        /// Initialize User binary conversion information.
        /// </summary>
        public static void InitializeUserBinaryConversionInfo()
        {
            // Get the base type of the converter info classes.
            Type baseClassType = typeof(BinaryConversionInfoBase);

            try
            {
                // Search for those who inherited from the DataModelValueConverterInfoBase.
                var q = from t in Assembly.GetExecutingAssembly().ExportedTypes
                        where t.IsPublic && t.IsClass && t.IsSubclassOf(baseClassType)
                        select t;

                // Run the static constructor of these classes.
                q.ForEach(t =>
                {
                    System.Runtime.CompilerServices.RuntimeHelpers.RunClassConstructor(t.TypeHandle);
                });
            }
            catch (ReflectionTypeLoadException ex)
            {
                Logger.Log(LogLevels.Debug, ex.Message);
                if (ex.LoaderExceptions != null)
                {
                    ex.LoaderExceptions.ForEach(le => Logger.Log(LogLevels.Debug, le.Message));
                }
            }

            // Register event handler for PropertyModified events from the data model proxy.
            DataModelProxy.PropertyModified += OnDataModelPropertyModified;
        }

        /// <summary>
        /// Handle PropertyModified event from the base data model.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event handler.</param>
        private static void OnDataModelPropertyModified(
            object sender,
            DataModelPropertyModifiedEventArgs e)
        {
            Guid guid;
            if (e.DataModel.GetPropertyGuid(e.PropertyName, out guid) == false)
            {
                // The property is not found, or the property has no Guid assigned to it.
                return;
            }

            // Find the binary field instances that has this input property.
            LinkedList<BinaryFieldInstance> list;
            if (propertyFieldMap.TryGetValue(guid, out list) == false)
            {
                // The binary field instance is not registered to the data model property.
                return;
            }

            SendModifiedBinaryContext.BeginSession();

            // Ask the binary field instances to update their binary data.
            var handledFields = new Dictionary<BinaryStructInstance, Queue<BinaryFieldInstance>>();
            foreach (BinaryFieldInstance field in list)
            {
                if (field.IsSourceProperty(e.DataModel, guid) == true)
                {
                    if (field.ConvertBinary() == false)
                    {
                        Logger.Log(LogLevels.Warning, "BinaryConversionInfoManager.OnDataModelPropertyModified : Failed converting binary data.");
                    }
                    else
                    {
                        SendModifiedBinaryContext.AddModifiedField(field);
                    }
                }
            }

            SendModifiedBinaryContext.EndSession();
        }
    }
}
