﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Reflection;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// A helper class for user data related operations.
    /// </summary>
    public class UserDataHelper
    {
        /// <summary>
        /// 唯一のインスタンスです。
        /// </summary>
        public static readonly UserDataHelper Instance = new UserDataHelper();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserDataHelper()
        {
            // ObsoleteUserDataConverter.dllを読み込む
            Assembly assembly = Assembly.LoadFrom(IOConstants.ObsoleteUserDataConverterDllPath);

            if (assembly == null)
            {
                Logger.Log(LogLevels.Error, "Failed to load ObsoleteUserDataConverter.dll.");
            }
            else
            {
                // ObsoleteUserDataConverterオブジェクトを作成
                var type = assembly.GetType("EffectMaker.ObsoleteUserDataConverter.ObsoleteUserDataConverter");
                this.ObsoleteUserDataConverter = Activator.CreateInstance(type) as IObsoleteUserDataConverter;

                if (this.ObsoleteUserDataConverter == null)
                {
                    Logger.Log(LogLevels.Error, "Failed to create ObsoleteUserDataConverter.");
                }
            }

            // Udd2Converter.dllを読み込む
            Assembly udd2Assembly = Assembly.LoadFrom(IOConstants.Udd2ConverterDllPath);

            if (udd2Assembly == null)
            {
                Logger.Log(LogLevels.Error, "Failed to load Udd2Converter.dll.");
            }
            else
            {
                // ObsoleteUserDataConverterオブジェクトを作成
                var type = udd2Assembly.GetType("EffectMaker.Udd2Converter.Udd2Converter");
                this.Udd2Converter = Activator.CreateInstance(type) as IUdd2Converter;

                if (this.Udd2Converter == null)
                {
                    Logger.Log(LogLevels.Error, "Failed to create Udd2Converter.");
                }
            }
        }

        /// <summary>
        /// USD, UDDコンバータを取得します。
        /// </summary>
        public IObsoleteUserDataConverter ObsoleteUserDataConverter { get; private set; }

        /// <summary>
        /// UDD2.0コンバータを取得します。
        /// </summary>
        public IUdd2Converter Udd2Converter { get; private set; }

        /// <summary>
        /// Load general user data.
        /// If setupForUse flag is set to true, set up the loaded definition
        /// and prepare the assemblies.
        /// </summary>
        /// <param name="defFilePath">The definition file path.</param>
        /// <param name="setupForUse">True to set up the loaded definition and prepare the assemblies.</param>
        /// <returns>The loaded definition.</returns>
        public UserDataDefinitionList LoadGeneralUserData(
            string defFilePath,
            bool setupForUse)
        {
            var loader = new UserDataDefinitionLoader();
            var def = loader.LoadDefinition(defFilePath);

            if (setupForUse == true)
            {
                loader.SetupUserData(def);
                UserDataManager.TriggerUserDataChangedEvent(UpdatedUserDataTypes.GeneralUserPage);
            }

            return def;
        }

        /// <summary>
        /// Load reserved shader user data.
        /// If setupForUse flag is set to true, set up the loaded definition
        /// and prepare the assemblies.
        /// </summary>
        /// <param name="defFilePath">The definition file path.</param>
        /// <param name="setupForUse">True to set up the loaded definition and prepare the assemblies.</param>
        /// <returns>The loaded definition.</returns>
        public ReservedShaderDefinitionList LoadReservedShaderUserData(
            string defFilePath,
            bool setupForUse)
        {
            var loader = new ReservedShaderDefinitionLoader();
            var def = loader.LoadDefinition(defFilePath);

            if (setupForUse == true)
            {
                loader.SetupUserData(def);
                UserDataManager.TriggerUserDataChangedEvent(UpdatedUserDataTypes.ReservedShader);
            }

            return def;
        }

        /// <summary>
        /// Load custom shader user data.
        /// If setupForUse flag is set to true, set up the loaded definition
        /// and prepare the assemblies.
        /// </summary>
        /// <param name="defFilePath">The definition file path.</param>
        /// <param name="setupForUse">True to set up the loaded definition and prepare the assemblies.</param>
        /// <returns>The loaded definition.</returns>
        public CustomShaderDefinitionList LoadCustomShaderUserData(
            string defFilePath,
            bool setupForUse)
        {
            var loader = new CustomShaderDefinitionLoader();
            var def = loader.LoadDefinition(defFilePath);
            var conversionInfo = loader.CollectObsoleteUserDataToConvert(def);

            // USDを新形式にコンバート
            UserDataManager.ConvertObsoleteUserData(conversionInfo, setupForUse);
            loader.PrepareConvertedObsoleteUserData(def);

            if (setupForUse == true)
            {
                loader.SetupUserData(def);
                UserDataManager.TriggerUserDataChangedEvent(UpdatedUserDataTypes.CustomShader);
            }

            return def;
        }

        /// <summary>
        /// Load custom action user data.
        /// If setupForUse flag is set to true, set up the loaded definition
        /// and prepare the assemblies.
        /// </summary>
        /// <param name="defFilePath">The definition file path.</param>
        /// <param name="setupForUse">True to set up the loaded definition and prepare the assemblies.</param>
        /// <returns>The loaded definition.</returns>
        public CustomActionDefinitionList LoadCustomActionUserData(
            string defFilePath,
            bool setupForUse)
        {
            var loader = new CustomActionDefinitionLoader();
            var def = loader.LoadDefinition(defFilePath);
            var conversionInfo = loader.CollectObsoleteUserDataToConvert(def);

            // UDDを新形式にコンバート
            UserDataManager.ConvertObsoleteUserData(conversionInfo, setupForUse);
            loader.PrepareConvertedObsoleteUserData(def);

            if (setupForUse == true)
            {
                loader.SetupUserData(def);
                UserDataManager.TriggerUserDataChangedEvent(UpdatedUserDataTypes.CustomAction);
            }

            return def;
        }

        /// <summary>
        /// エミッタ拡張パラメータのユーザ定義を読み込みます。
        /// setupForUseフラグを立てることで、定義の読み込みに加えてアセンブリの準備なども行います。
        /// </summary>
        /// <param name="defFilePath">The definition file path.</param>
        /// <param name="setupForUse">True to set up the loaded definition and prepare the assemblies.</param>
        /// <returns>The loaded definition.</returns>
        public EmitterExtParamsDefinition LoadEmitterExtParamsUserData(
            string defFilePath,
            bool setupForUse)
        {
            var loader = new EmitterExtParamsDefinitionLoader();
            var def = loader.LoadDefinition(defFilePath);

            if (setupForUse == true)
            {
                loader.SetupUserData(def);
                UserDataManager.TriggerUserDataChangedEvent(UpdatedUserDataTypes.EmitterExtParams);
            }

            return def;
        }

        /// <summary>
        /// Load and set up custom shader and custom action for use.
        /// </summary>
        /// <param name="customShaderDefPath">The definition file path for the custom shader.</param>
        /// <param name="customActionDefPath">The definition file path for the custom action.</param>
        /// <returns>True on success.</returns>
        public bool LoadAndSetupCustomShaderAndAction(
            string customShaderDefPath,
            string customActionDefPath)
        {
            // Load custom shader.
            var customShaderLoader = new CustomShaderDefinitionLoader();
            var customShaderDef = customShaderLoader.LoadDefinition(customShaderDefPath);

            var customShaderConversionInfo =
                customShaderLoader.CollectObsoleteUserDataToConvert(customShaderDef);

            // Load custom action.
            var customActionLoader = new CustomActionDefinitionLoader();
            var customActionDef = customActionLoader.LoadDefinition(customActionDefPath);

            var customActionConversionInfo =
                customActionLoader.CollectObsoleteUserDataToConvert(customActionDef);

            // Convert .udd or .usd files.
            UserDataManager.ConvertObsoleteUserData(
                customShaderConversionInfo.Concat(customActionConversionInfo),
                true);

            // Prepare the .udd/.usd definitions for use.
            customShaderLoader.PrepareConvertedObsoleteUserData(customShaderDef);
            customActionLoader.PrepareConvertedObsoleteUserData(customActionDef);

            // Set up the user data for use.
            customShaderLoader.SetupUserData(customShaderDef);
            customActionLoader.SetupUserData(customActionDef);

            UserDataManager.TriggerUserDataChangedEvent(UpdatedUserDataTypes.CustomShaderAndAction);

            return true;
        }

        /// <summary>
        /// Load and set up all the user data for use.
        /// </summary>
        /// <param name="generalUserDataDefPath">The definition file path for the general user data.</param>
        /// <param name="customShaderDefPath">The definition file path for the custom shader.</param>
        /// <param name="customActionDefPath">The definition file path for the custom action.</param>
        /// <param name="emitterExtParamsDefPath">The definition file path for the emitter extension parameters.</param>
        /// <param name="reservedShaderDefPath">The definition file path for the reserved shader.</param>
        /// <returns>True on success.</returns>
        public bool LoadAndSetupAllUserData(
            string generalUserDataDefPath,
            string customShaderDefPath,
            string customActionDefPath,
            string emitterExtParamsDefPath,
            string reservedShaderDefPath)
        {
            // Load custom shader.
            var customShaderLoader = new CustomShaderDefinitionLoader();
            var customShaderDef = customShaderLoader.LoadDefinition(customShaderDefPath);

            var customShaderConversionInfo =
                customShaderLoader.CollectObsoleteUserDataToConvert(customShaderDef);

            // Load custom action.
            var customActionLoader = new CustomActionDefinitionLoader();
            var customActionDef = customActionLoader.LoadDefinition(customActionDefPath);

            var customActionConversionInfo =
                customActionLoader.CollectObsoleteUserDataToConvert(customActionDef);

            var emitterExtParamsLoader = new EmitterExtParamsDefinitionLoader();
            var emitterExtParamsDef = emitterExtParamsLoader.LoadDefinition(emitterExtParamsDefPath);

            // Load reserved shader.
            var reservedShaderLoader = new ReservedShaderDefinitionLoader();
            var reservedShaderDef = reservedShaderLoader.LoadDefinition(reservedShaderDefPath);

            // Load general user data.
            var userDataLoader = new UserDataDefinitionLoader();
            var userDataDef = userDataLoader.LoadDefinition(generalUserDataDefPath);

            // Convert .udd or .usd files.
            UserDataManager.ConvertObsoleteUserData(
                customShaderConversionInfo.Concat(customActionConversionInfo),
                true);

            // Prepare the .udd/.usd definitions for use.
            customShaderLoader.PrepareConvertedObsoleteUserData(customShaderDef);
            customActionLoader.PrepareConvertedObsoleteUserData(customActionDef);

            // Set up the user data for use.
            customShaderLoader.SetupUserData(customShaderDef);
            customActionLoader.SetupUserData(customActionDef);
            emitterExtParamsLoader.SetupUserData(emitterExtParamsDef);
            reservedShaderLoader.SetupUserData(reservedShaderDef);
            userDataLoader.SetupUserData(userDataDef);

            UserDataManager.TriggerUserDataChangedEvent(UpdatedUserDataTypes.All);

            return true;
        }
    }
}
