﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml;
using System.Xml.Serialization;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Data class that holds custom shader definitions.
    /// </summary>
    public class CustomShaderDefinitionList : IXmlDocSerializable
    {
        /// <summary>The path where this definition is loaded from.</summary>
        private string filePath = string.Empty;

        /// <summary>The base folder path.</summary>
        private string baseFolderPath = string.Empty;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public CustomShaderDefinitionList()
        {
            this.GeneralShaderDefinition = new GeneralShaderDefinition();
            this.CustomShaderDefinitions = new List<CustomShaderDefinition>();
        }

        /// <summary>
        /// The base folder path of the definitions.
        /// </summary>
        public string BaseFolderPath
        {
            get
            {
                return this.baseFolderPath;
            }

            private set
            {
                this.baseFolderPath = PathUtility.LoadEnvironments(value);
                this.GeneralShaderDefinition.BaseFolderPath = this.baseFolderPath;
                this.CustomShaderDefinitions.ForEach(s => s.BaseFolderPath = this.baseFolderPath);
            }
        }

        /// <summary>
        /// Get or set the file path where this definition was loaded from.
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }

            set
            {
                this.filePath = value;
                this.BaseFolderPath = Path.GetDirectoryName(this.filePath);
            }
        }

        /// <summary>
        /// The general shader definition.
        /// </summary>
        public GeneralShaderDefinition GeneralShaderDefinition { get; set; }

        /// <summary>
        /// The custom shader definitions.
        /// </summary>
        public List<CustomShaderDefinition> CustomShaderDefinitions { get; set; }

        /// <summary>
        /// Enumerates the old custom shader definitions in the .usd files.
        /// </summary>
        public IEnumerable<ObsoleteCustomShaderDefinition> UsdDefinitions
        {
            get
            {
                return from def in this.CustomShaderDefinitions
                       where def is ObsoleteCustomShaderDefinition
                       select (ObsoleteCustomShaderDefinition)def;
            }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.GeneralShaderDefinition =
                this.ReadElement<GeneralShaderDefinition>(context, "GeneralShaders", null);
            if (this.GeneralShaderDefinition == null)
            {
                this.GeneralShaderDefinition =
                    this.ReadElement<GeneralShaderDefinition>(context, "GeneralCustomShaders", null);
                if (this.GeneralShaderDefinition == null)
                {
                    this.GeneralShaderDefinition = new GeneralShaderDefinition();
                }
            }

            var nameTypeMap = new Dictionary<string, Type>()
            {
                { "DefLinkData", typeof(CustomShaderDefinition) },
                { "CustomShaderUIDefLinkData", typeof(ObsoleteCustomShaderDefinition) },
            };

            this.CustomShaderDefinitions = this.ReadElementsByTagNames<CustomShaderDefinition>(
                context,
                nameTypeMap).ToList();

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }

        /// <summary>
        /// Load the custom shader definition file with the link path.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool LoadCustomShaderDefinitionFiles()
        {
            bool result = true;
            foreach (CustomShaderDefinition link in this.CustomShaderDefinitions)
            {
                if ((link is ObsoleteCustomShaderDefinition) == false)
                {
                    result &= link.LoadCustomShaderDefinitionFile();
                }
            }

            return result;
        }
    }

    /// <summary>
    /// Data class that holds definition of the general shader.
    /// </summary>
    [Serializable]
    public class GeneralShaderDefinition : IXmlDocSerializable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public GeneralShaderDefinition()
        {
            this.VertexShaderPaths = new List<string>();
            this.FragmentShaderPaths = new List<string>();
            this.ComputeShaderPaths = new List<string>();
        }

        /// <summary>
        /// The base folder path of the definitions.
        /// </summary>
        public string BaseFolderPath { get; set; }

        /// <summary>
        /// 頂点シェーダのパスリストを取得または設定します。
        /// </summary>
        public List<string> VertexShaderPaths { get; set; }

        /// <summary>
        /// 頂点シェーダのフルパスを列挙します。
        /// </summary>
        public IEnumerable<string> VertexShaderFullPaths
        {
            get
            {
                return this.VertexShaderPaths.Select(x => PathUtility.ToAbsolutePath(x, this.BaseFolderPath));
            }
        }

        /// <summary>
        /// フラグメントシェーダのパスリストを取得または設定します。
        /// </summary>
        public List<string> FragmentShaderPaths { get; set; }

        /// <summary>
        /// フラグメントシェーダのフルパスを列挙します。
        /// </summary>
        public IEnumerable<string> FragmentShaderFullPaths
        {
            get
            {
                return this.FragmentShaderPaths.Select(x => PathUtility.ToAbsolutePath(x, this.BaseFolderPath));
            }
        }

        /// <summary>
        /// コンピュートシェーダのパスリストを取得または設定します。
        /// </summary>
        public List<string> ComputeShaderPaths { get; set; }


        /// <summary>
        /// コンピュートシェーダのフルパスを列挙します。
        /// </summary>
        public IEnumerable<string> ComputeShaderFullPaths
        {
            get
            {
                return this.ComputeShaderPaths.Select(x => PathUtility.ToAbsolutePath(x, this.BaseFolderPath));
            }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool ReadXml(XmlDocSerializationContext context)
        {
            this.VertexShaderPaths = this.ReadListElement<string>(
                context,
                "VertexShaderPaths",
                this.VertexShaderPaths);

            this.FragmentShaderPaths = this.ReadListElement<string>(
                context,
                "FragmentShaderPaths",
                this.FragmentShaderPaths);

            this.ComputeShaderPaths = this.ReadListElement<string>(
                context,
                "ComputeShaderPaths",
                this.ComputeShaderPaths);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }
    }

    /// <summary>
    /// Data class that holds informations of a custom shader.
    /// </summary>
    public class CustomShaderDefinition : IXmlDocSerializable
    {
        /// <summary>The relative path of the definition file.</summary>
        private string definitionRelativePath = string.Empty;

        /// <summary>The custom shader definition.</summary>
        private InternalCustomShaderDefinition internalDefinition = null;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public CustomShaderDefinition()
        {
        }

        /// <summary>
        /// The base folder path of the definitions.
        /// </summary>
        public string BaseFolderPath { get; set; }

        /// <summary>
        /// The ID of the custom shader.
        /// </summary>
        public int Id { get; set; }

        /// <summary>
        /// The file path to the custom shader definition.
        /// </summary>
        public string DefinitionPath
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.definitionRelativePath); }
            set { this.definitionRelativePath = value; }
        }

        /// <summary>
        /// Get the flag indicating whether the definition is correctly loaded.
        /// </summary>
        public bool IsDefinitionLoaded
        {
            get { return this.internalDefinition != null; }
        }

        /// <summary>
        /// The display name for the custom shader.
        /// </summary>
        public string Name
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.Name : string.Empty; }
        }

        /// <summary>
        /// The English display name for the custom shader.
        /// </summary>
        public string NameEn
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.NameEn : string.Empty; }
        }

        /// <summary>
        /// The source file path of the data model.
        /// </summary>
        public string DataModelFilePath
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.DataModelFullFilePath : string.Empty; }
        }

        /// <summary>
        /// The XAML file name for the custom shader UI.
        /// </summary>
        public string XamlFilePath
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.XamlFullFilePath : string.Empty; }
        }

        /// <summary>
        /// The source file path of the binary conversion info.
        /// </summary>
        public string BinaryConversionInfoPath
        {
            get { return this.IsDefinitionLoaded ? this.internalDefinition.BinaryConversionInfoFullPath : string.Empty; }
        }

        /// <summary>
        /// 頂点シェーダのフルパスを列挙します。
        /// </summary>
        public IEnumerable<string> VertexShaderFullPaths
        {
            get
            {
                if (this.IsDefinitionLoaded == false)
                {
                    return Enumerable.Empty<string>();
                }

                return this.internalDefinition.VertexShaderFullPaths;
            }
        }

        /// <summary>
        /// フラグメントシェーダのフルパスを列挙します。
        /// </summary>
        public IEnumerable<string> FragmentShaderFullPaths
        {
            get
            {
                if (this.IsDefinitionLoaded == false)
                {
                    return Enumerable.Empty<string>();
                }

                return this.internalDefinition.FragmentShaderFullPaths;
            }
        }

        /// <summary>
        /// コンピュートシェーダのフルパスを列挙します。
        /// </summary>
        public IEnumerable<string> ComputeShaderFullPaths
        {
            get
            {
                if (this.IsDefinitionLoaded == false)
                {
                    return Enumerable.Empty<string>();
                }

                return this.internalDefinition.ComputeShaderFullPaths;
            }
        }

        /// <summary>
        /// UDD2.0を使用した定義か否かを取得します。
        /// </summary>
        public bool UsingUdd2
        {
            get { return this.internalDefinition != null && this.internalDefinition.UsingUdd2; }
        }

        /// <summary>
        /// The full path of the Udd2 file.
        /// </summary>
        public string Udd2FullFilePath
        {
            get { return this.internalDefinition != null ? this.internalDefinition.Udd2FullFilePath : string.Empty; }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool ReadXml(XmlDocSerializationContext context)
        {
            this.Id = this.ReadAttribute(context, "SettingId", this.Id);
            this.Id = this.ReadAttribute(context, "CallbackId", this.Id);

            // Read the file path of the actual custom shader definition.
            this.definitionRelativePath = context.CurrentNode.InnerText;

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }

        /// <summary>
        /// Load the custom shader definition file with the link path.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool LoadCustomShaderDefinitionFile()
        {
            if (File.Exists(this.DefinitionPath) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "Custom shader definition does not exist. ({0})",
                    this.DefinitionPath);
                return false;
            }

            try
            {
                InternalCustomShaderDefinition def =
                    SerializationHelper.LoadXmlDocSerializable<InternalCustomShaderDefinition>(this.DefinitionPath);

                def.BaseFolderPath = Path.GetDirectoryName(this.DefinitionPath);
                if (def.UsingUdd2)
                {
                    def.Name = UserDataHelper.Instance.Udd2Converter.LoadDefinition(def.Udd2FullFilePath)
                        ? UserDataHelper.Instance.Udd2Converter.DefineName : def.Name;
                }

                this.internalDefinition = def;
            }
            catch
            {
                Logger.Log(
                    LogLevels.Warning,
                    "Failed loading custom shader definition {0}",
                    this.DefinitionPath);
                return false;
            }

            return true;
        }

        /// <summary>
        /// Data class that holds informations of a custom shader.
        /// </summary>
        private class InternalCustomShaderDefinition : GeneralShaderDefinition
        {
            /// <summary>
            /// Default constructor.
            /// </summary>
            public InternalCustomShaderDefinition()
            {
            }

            /// <summary>
            /// UDD2.0を使用した定義か否かを取得します。
            /// </summary>
            [XmlIgnore]
            public bool UsingUdd2 { get; private set; }

            /// <summary>
            /// The Udd2 file name. (can be a relative path)
            /// </summary>
            public string Udd2FileName { get; set; }

            /// <summary>
            /// The full path of the Udd2 file.
            /// </summary>
            public string Udd2FullFilePath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.Udd2FileName); }
            }

            /// <summary>
            /// The display name for the custom shader.
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// The English display name for the custom shader.
            /// </summary>
            public string NameEn { get; set; }

            /// <summary>
            /// The source file path of the data model.
            /// </summary>
            public string DataModelFilePath { get; set; }

            /// <summary>
            /// The full path of the data model source file.
            /// </summary>
            public string DataModelFullFilePath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.DataModelFilePath); }
            }

            /// <summary>
            /// The XAML file name for the custom shader UI.
            /// </summary>
            public string XamlFilePath { get; set; }

            /// <summary>
            /// The full path of the XAML file for the custom action UI.
            /// </summary>
            public string XamlFullFilePath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.XamlFilePath); }
            }

            /// <summary>
            /// The source file path of the binary conversion info.
            /// </summary>
            public string BinaryConversionInfoPath { get; set; }

            /// <summary>
            /// The full path to the binary conversion info source file.
            /// </summary>
            public string BinaryConversionInfoFullPath
            {
                get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.BinaryConversionInfoPath); }
            }

            /// <summary>
            /// Deserializes from the given XML node.
            /// </summary>
            /// <param name="context">The data context needed for the deserialization.</param>
            /// <returns>True on success.</returns>
            public override bool ReadXml(XmlDocSerializationContext context)
            {
                if (base.ReadXml(context) == false)
                {
                    return false;
                }

                this.Name = this.ReadAttribute(context, "Label", this.Name);
                this.NameEn = this.ReadAttribute(context, "Label_EN", this.NameEn);

                var node = XmlDocSerializationHelper.EnumerateElementsByTagName(
                    context.CurrentNode,
                    "DataModelSetting").FirstOrDefault();

                if (node != null)
                {
                    context.PushCurrentNode(node);

                    this.Udd2FileName =
                        this.ReadAttribute(context, "DefinitionFile", this.Udd2FileName);
                    if (string.IsNullOrEmpty(this.Udd2FileName))
                    {
                        this.Udd2FileName =
                            this.ReadAttribute(context, "Udd2Definition", this.Udd2FileName);
                    }

                    if (!string.IsNullOrEmpty(this.Udd2FileName))
                    {
                        this.UsingUdd2 = true;
                        return true;
                    }

                    this.DataModelFilePath =
                        this.ReadAttribute(context, "DataModel", this.DataModelFilePath);

                    this.XamlFilePath =
                        this.ReadAttribute(context, "UI", this.XamlFilePath);

                    this.BinaryConversionInfoPath =
                        this.ReadAttribute(context, "ConversionInfo", this.BinaryConversionInfoPath);

                    context.PopCurrentNode();
                }

                return true;
            }
        }
    }

    /// <summary>
    /// Custom shader info that reads old custom shader link data.
    /// (converts .usd files to new custom shader info)
    /// </summary>
    public class ObsoleteCustomShaderDefinition : CustomShaderDefinition
    {
        /// <summary>The relative path of the .usd file.</summary>
        private string usdRelativePath = string.Empty;

        /// <summary>
        /// Constructor.
        /// </summary>
        public ObsoleteCustomShaderDefinition()
        {
        }

        /// <summary>
        /// Get or set the .usd file path.
        /// </summary>
        public string UsdFilePath
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.usdRelativePath); }
            set { this.usdRelativePath = PathUtility.LoadEnvironments(value); }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.Id = this.ReadAttribute(context, "CallbackId", this.Id);
            this.Id = this.ReadAttribute(context, "SettingId", this.Id);

            // Read the file path of the actual custom shader definition.
            this.UsdFilePath = context.CurrentNode.InnerText;
            this.DefinitionPath = this.UsdFilePath;

            return true;
        }
    }
}
