﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using SerializeResult = EffectMaker.BusinessLogic.Serializer.DataModelSerializer.SerializeResult;
using DeserializeResult = EffectMaker.BusinessLogic.Serializer.DataModelSerializer.DeserializeResult;

namespace EffectMaker.BusinessLogic.Serializer
{
    /// <summary>
    /// データモデルのシリアライズユーティリティです。
    /// 主に、ワークスペースノードデータの複製・コピーに使用します。
    /// </summary>
    public static class DataModelSerializeUtility
    {
        /// <summary>
        /// シリアライザリストです。
        /// </summary>
        private static Dictionary<string, DataModelSerializer> serializerList;

        /// <summary>
        /// シリアライザリストを取得します。
        /// </summary>
        private static Dictionary<string, DataModelSerializer> SerializerList
        {
            get
            {
                if (serializerList == null)
                {
                    DataModelSerializer emitterDataSerializer            = new DataModelSerializer<EmitterData           >();
                    DataModelSerializer emitterSetDataSerializer         = new EmitterSetDataSerializer();
                    DataModelSerializer previewDataSerializer            = new PreviewDataSerializer();
                    DataModelSerializer fieldDataSerializer              = new DataModelSerializer<FieldDataBase         >();
                    DataModelSerializer stripeDataSerializer             = new DataModelSerializer<StripeData            >();
                    DataModelSerializer reservedShaderNodeDataSerializer = new DataModelSerializer<ReservedShaderNodeData>();
                    DataModelSerializer customActionDataSerializer       = new DataModelSerializer<CustomActionData      >();
                    DataModelSerializer emitterExtParamsSerializer       = new DataModelSerializer<EmitterExtParams      >();
                    // 未対応
                    // DataModelSerializer workspaceDataType                = new DataModelSerializer<WorkspaceData         >();

                    serializerList = new Dictionary<string, DataModelSerializer>
                    {
                        { typeof(EmitterData           ).FullName, emitterDataSerializer            },
                        { typeof(EmitterSetData        ).FullName, emitterSetDataSerializer         },
                        { typeof(PreviewData           ).FullName, previewDataSerializer            },
                        { typeof(AddLocationData       ).FullName, fieldDataSerializer              },
                        { typeof(CollisionData         ).FullName, fieldDataSerializer              },
                        { typeof(ConvergeData          ).FullName, fieldDataSerializer              },
                        { typeof(CurlNoiseData         ).FullName, fieldDataSerializer              },
                        { typeof(MagnetData            ).FullName, fieldDataSerializer              },
                        { typeof(RandomData            ).FullName, fieldDataSerializer              },
                        { typeof(RandomFe1Data         ).FullName, fieldDataSerializer              },
                        { typeof(SpinData              ).FullName, fieldDataSerializer              },
                        { typeof(CustomFieldData       ).FullName, fieldDataSerializer              },
                        { typeof(StripeHistoryData     ).FullName, stripeDataSerializer             },
                        { typeof(StripeComplexData     ).FullName, stripeDataSerializer             },
                        { typeof(StripeSuperData       ).FullName, stripeDataSerializer             },
                        { typeof(ReservedShaderNodeData).FullName, reservedShaderNodeDataSerializer },
                        { typeof(CustomActionData      ).FullName, customActionDataSerializer       },
                        { typeof(EmitterExtParams      ).FullName, emitterExtParamsSerializer       },
                        // 未対応
                        // { typeof(WorkspaceData         ).FullName, workspaceDataSerializer          },
                    };
                }

                return serializerList;
            }
        }

        /// <summary>
        /// シリアライズを行います。
        /// </summary>
        /// <param name="dataModel">シリアライズする値</param>
        /// <param name="stream">出力ストリーム</param>
        /// <returns>シリアライズ結果を返します。</returns>
        public static SerializeResult Serialize(DataModelBase dataModel, Stream stream)
        {
            // 型名からシリアライザを検索
            DataModelSerializer serializer;
            string typeFullName = dataModel.GetType().FullName;
            SerializerList.TryGetValue(typeFullName, out serializer);

            if (serializer == null)
            {
                SerializeResult result = new SerializeResult();
                result.Exception = new InvalidOperationException ("Serializer not found.");

                return result;
            }

            // シリアライズを実行
            return serializer.Serialize(dataModel, stream);
        }

        /// <summary>
        /// デシリアライズを行います。
        /// </summary>
        /// <param name="value">デシリアライズする値</param>
        /// <param name="typeFullName">デシリアライズする値の型名</param>
        /// <returns>デシリアライズ結果を返します。</returns>
        public static DeserializeResult Deserialize(Stream stream, string typeFullName)
        {
            // 型名からシリアライザを検索
            DataModelSerializer serializer;
            SerializerList.TryGetValue(typeFullName, out serializer);

            if (serializer == null)
            {
                DeserializeResult result = new DeserializeResult();
                result.Exception = new InvalidOperationException("Serializer not found.");

                return result;
            }

            // デシリアライズを実行
            return serializer.Deserialize(stream);
        }

        /// <summary>
        /// クリップボードデータへのシリアライズを行います。
        /// </summary>
        /// <param name="dataModel">シリアライズする値</param>
        /// <param name="data">クリップボードデータ</param>
        public static SerializeResult SerializeToClipboardData(DataModelBase dataModel, out string data)
        {
            SerializeResult result;

            using (MemoryStream stream = new MemoryStream())
            {
                result = Serialize(dataModel, stream);

                if (result.IsSuccess == false)
                {
                    data = null;
                    return result;
                }

                byte[] byteData = new byte[stream.Length];

                stream.Position = 0;
                stream.Read(byteData, 0, byteData.Length);

                data = Encoding.UTF8.GetString(byteData);
            }

            return result;
        }

        /// <summary>
        /// クリップボードデータからのデシリアライズを行います。
        /// </summary>
        /// <param name="data">クリップボードデータ</param>
        /// <param name="typeFullName">デシリアライズする値の型名</param>
        /// <returns>デシリアライズ結果を返します。</returns>
        public static DeserializeResult DeserializeFromClipboardData(string data, string typeFullName)
        {
            byte[] byteData = Encoding.UTF8.GetBytes(data);

            DeserializeResult result;

            using (MemoryStream stream = new MemoryStream(byteData))
            {
                result = Deserialize(stream, typeFullName);
            }

            return result;
        }
    }
}
