﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Serialization;

using EffectMaker.BusinessLogic.IO;
using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// Basic options.
    /// </summary>
    public class BasicOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the BasicOptions instance.
        /// </summary>
        public BasicOptions()
        {
            this.Undo = new UndoOptions();
            this.Emitters = new BasicEmittersOptions();
            this.Textures = new BasicTexturesOptions();
            this.Primitives = new BasicPrimitivesOptions();
            this.Preview = new BasicPreviewOptions();
            this.DocumentPath = IOConstants.DefaultDocumentPath;
        }

        /// <summary>
        /// Copy the BasicOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public BasicOptions(BasicOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Gets or sets the UndoOptions instance.
        /// </summary>
        public UndoOptions Undo { get; set; }

        /// <summary>
        /// Gets the workspace path for files to auto-load.
        /// </summary>
        public string WorkspaceFilesAutoLoad { get; set; }

        /// <summary>
        /// Gets or sets the basic textures options.
        /// </summary>
        public BasicEmittersOptions Emitters { get; set; }

        /// <summary>
        /// Gets or sets the basic textures options.
        /// </summary>
        public BasicTexturesOptions Textures { get; set; }

        /// <summary>
        /// Gets or sets the basic primitives options.
        /// </summary>
        public BasicPrimitivesOptions Primitives { get; set; }

        /// <summary>
        /// Gets or sets the basic primitives options.
        /// </summary>
        public BasicPreviewOptions Preview { get; set; }

        /// <summary>
        /// ドキュメントの起動パスを取得または設定します。
        /// </summary>
        public string DocumentPath { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as BasicOptions;
            if (options == null)
            {
                return false;
            }

            this.Undo.Set(options.Undo);
            this.WorkspaceFilesAutoLoad = options.WorkspaceFilesAutoLoad;
            this.Emitters.Set(options.Emitters);
            this.Textures.Set(options.Textures);
            this.Primitives.Set(options.Primitives);
            this.Preview.Set(options.Preview);
            this.DocumentPath = options.DocumentPath;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.Undo = this.ReadElement(context, "Undo", this.Undo);

            this.WorkspaceFilesAutoLoad = this.ReadElement(
                context,
                "WorkspaceFilesAutoLoad",
                this.WorkspaceFilesAutoLoad);

            this.Emitters = this.ReadElement(
                context,
                "Emitters",
                this.Emitters);

            this.Textures = this.ReadElement(
                context,
                "Textures",
                this.Textures);

            this.Primitives = this.ReadElement(
                context,
                "Primitives",
                this.Primitives);

            this.Preview = this.ReadElement(
                context,
                "Preview",
                this.Preview);

            this.DocumentPath = this.ReadElement(context, "DocumentPath", this.DocumentPath);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "Undo", this.Undo);
            this.WriteElement(context, "WorkspaceFilesAutoLoad", this.WorkspaceFilesAutoLoad);
            this.WriteElement(context, "Emitters", this.Emitters);
            this.WriteElement(context, "Textures", this.Textures);
            this.WriteElement(context, "Primitives", this.Primitives);
            this.WriteElement(context, "Preview", this.Preview);
            this.WriteElement(context, "DocumentPath", this.DocumentPath);

            return true;
        }
    }

    /// <summary>
    /// Undo options.
    /// </summary>
    public class UndoOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the UndoOptions instance.
        /// </summary>
        public UndoOptions()
        {
            this.MaximumStackItems = 100;
        }

        /// <summary>
        /// Copy the UndoOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public UndoOptions(UndoOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Gets or sets the undo/redo availability.
        /// </summary>
        [XmlAttribute]
        public bool IsEnabled { get; set; }

        /// <summary>
        /// Gets or sets the maximum number of undo items.
        /// </summary>
        [XmlAttribute]
        public int MaximumStackItems { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as UndoOptions;
            if (options == null)
            {
                return false;
            }

            this.IsEnabled = options.IsEnabled;
            this.MaximumStackItems = options.MaximumStackItems;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.IsEnabled = this.ReadAttribute(context, "IsEnabled", this.IsEnabled);
            this.MaximumStackItems = this.ReadAttribute(context, "MaximumStackItems", this.MaximumStackItems);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "IsEnabled", this.IsEnabled);
            this.WriteAttribute(context, "MaximumStackItems", this.MaximumStackItems);

            return true;
        }
    }

    /// <summary>
    /// Basic texture options.
    /// </summary>
    public class BasicEmittersOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the BasicEmittersOptions instance.
        /// </summary>
        public BasicEmittersOptions()
        {
            this.ShowWarningIconLinearTexture = true;
        }

        /// <summary>
        /// Copy the BasicEmittersOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public BasicEmittersOptions(BasicEmittersOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// リニア編集モードの設定と異なるテクスチャが設定されているエミッタに警告マークを表示するかどうか取得または設定します。
        /// </summary>
        [XmlAttribute]
        public bool ShowWarningIconLinearTexture { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as BasicEmittersOptions;
            if (options == null)
            {
                return false;
            }

            this.ShowWarningIconLinearTexture = options.ShowWarningIconLinearTexture;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.ShowWarningIconLinearTexture = this.ReadAttribute(context, "ShowWarningIconLinearTexture", this.ShowWarningIconLinearTexture);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "ShowWarningIconLinearTexture", this.ShowWarningIconLinearTexture);

            return true;
        }
    }

    /// <summary>
    /// Basic texture options.
    /// </summary>
    public class BasicTexturesOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the BasicTexturesOptions instance.
        /// </summary>
        public BasicTexturesOptions()
        {
            this.ShowWarning = true;
            this.AutoReload = true;
        }

        /// <summary>
        /// Copy the BasicTexturesOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public BasicTexturesOptions(BasicTexturesOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Show warning on loading when textures are not accessible.
        /// </summary>
        [XmlAttribute]
        public bool ShowWarning { get; set; }

        /// <summary>
        /// Automatically reload when referencing textures are changed.
        /// </summary>
        [XmlAttribute]
        public bool AutoReload { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as BasicTexturesOptions;
            if (options == null)
            {
                return false;
            }

            this.ShowWarning = options.ShowWarning;
            this.AutoReload = options.AutoReload;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.ShowWarning = this.ReadAttribute(context, "ShowWarning", this.ShowWarning);
            this.AutoReload = this.ReadAttribute(context, "AutoReload", this.AutoReload);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "ShowWarning", this.ShowWarning);
            this.WriteAttribute(context, "AutoReload", this.AutoReload);

            return true;
        }
    }

    /// <summary>
    /// Basic primitives options.
    /// </summary>
    public class BasicPrimitivesOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the BasicPrimitivesOptions instance.
        /// </summary>
        public BasicPrimitivesOptions()
        {
            this.ShowWarning = true;
            this.AutoReload = true;
        }

        /// <summary>
        /// Copy the BasicPrimitivesOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public BasicPrimitivesOptions(BasicPrimitivesOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Show warning when loading inaccessible primitives.
        /// </summary>
        [XmlAttribute]
        public bool ShowWarning { get; set; }

        /// <summary>
        /// Automatically reload when referencing primitives are changed.
        /// </summary>
        [XmlAttribute]
        public bool AutoReload { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as BasicPrimitivesOptions;
            if (options == null)
            {
                return false;
            }

            this.ShowWarning = options.ShowWarning;
            this.AutoReload = options.AutoReload;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.ShowWarning = this.ReadAttribute(context, "ShowWarning", this.ShowWarning);
            this.AutoReload = this.ReadAttribute(context, "AutoReload", this.AutoReload);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "ShowWarning", this.ShowWarning);
            this.WriteAttribute(context, "AutoReload", this.AutoReload);

            return true;
        }
    }

    /// <summary>
    /// Basic preview options.
    /// </summary>
    public class BasicPreviewOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the BasicPreviewOptions instance.
        /// </summary>
        public BasicPreviewOptions()
        {
            this.IncludeAllSave = false;
        }

        /// <summary>
        /// Copy the BasicPreviewOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public BasicPreviewOptions(BasicPrimitivesOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// 全て上書き保存でプレビューも保存するか否か.
        /// </summary>
        [XmlAttribute]
        public bool IncludeAllSave { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as BasicPreviewOptions;
            if (options == null)
            {
                return false;
            }

            this.IncludeAllSave = options.IncludeAllSave;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.IncludeAllSave = this.ReadAttribute(context, "IncludeAllSave", this.IncludeAllSave);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "IncludeAllSave", this.IncludeAllSave);

            return true;
        }
    }
}
