﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.BusinessLogic.Options;

using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.IO
{
    /// <summary>
    /// ApplicationIOManager.
    /// </summary>
    public static class ApplicationIOManager
    {
        /// <summary>
        /// 最後にアクセスしたディレクトリを取得します。
        /// </summary>
        /// <param name="type">タイプ</param>
        /// <returns>ディレクトリパスを返します。</returns>
        public static string GetLastAccessedDir(LastAccessDirectoryTypes type)
        {
            var directories = OptionStore.RootOptions.FileEvent.LastAccessedDirectories;

            // パスを取得
            string path = null;
            directories.TryGetValue(type, out path);

            // パスが登録されていないとき、ドキュメントのフォルダパスを返す
            if (path == null)
            {
                path = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
            }

            return path;
        }

        /// <summary>
        /// 最後にアクセスしたディレクトリを設定します。
        /// </summary>
        /// <param name="type">タイプ</param>
        /// <param name="path">ディレクトリにあるファイルのパス</param>
        public static void SetLastAccessedDir(LastAccessDirectoryTypes type, string path)
        {
            var directories = OptionStore.RootOptions.FileEvent.LastAccessedDirectories;

            // パスに入っているファイル名を削る
            string dirPath = PathUtility.GetDirectoryName(path) + "\\";

            // パスを設定
            directories[type] = dirPath;
        }

        /// <summary>
        /// Open text file with external text editor that is set in the config.
        /// </summary>
        /// <param name="filePath">The file path of the text file.</param>
        /// <returns>True on success.</returns>
        public static bool OpenTextFileWithExternalTextEditor(string filePath)
        {
            // The log destinations.
            string logDest = "LogView,DebugConsole";

            // External text editor path.
            string editorPath = OptionStore.RootOptions.Interface.TextEditorPath;

            // Decide the text editor to use.
            if (string.IsNullOrEmpty(editorPath) == true)
            {
                // External text editor was not set, use Notepad instead.
                editorPath = System.Environment.SystemDirectory + "\\notepad.exe";
                if (File.Exists(editorPath) == false)
                {
                    string str = Properties.Resources.WarningNotePadNotFound;
                    Logger.Log(logDest, LogLevels.Warning, str);
                    return false;
                }
            }
            else if (File.Exists(editorPath) == false)
            {
                // The assigned text editor executable does not exist.
                string str = Properties.Resources.WarningExternalTextEditorNotFound;
                Logger.Log(logDest, LogLevels.Warning, str, editorPath);

                // Use Notepad by default.
                editorPath = System.Environment.SystemDirectory + "\\notepad.exe";
                if (File.Exists(editorPath) == false)
                {
                    str = Properties.Resources.WarningNotePadNotFound;
                    Logger.Log(logDest, LogLevels.Warning, str);
                    return false;
                }
            }

            try
            {
                // Launch external text editor to show the shader.
                Process textEditorProcess = Process.Start(editorPath, filePath);

                if (textEditorProcess == null)
                {
                    string str = Properties.Resources.WarningFailedLaunchTextEditor;
                    Logger.Log(logDest, LogLevels.Warning, str);
                    return false;
                }
            }
            catch (System.Exception ex)
            {
                string str = Properties.Resources.WarningFailedLaunchTextEditor;
                Logger.Log(logDest, LogLevels.Warning, str);
                Logger.Log(logDest, LogLevels.Warning, ex.Message);
                return false;
            }

            return true;
        }
    }
}
