﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.EffectCombinerEditor;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Manager;
using EffectMaker.DataModel.Specific.DataModels;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Shader
{
    /// <summary>
    /// Write shader array to binary stream.
    /// </summary>
    public class ShaderArrayWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// The shader types to add.
        /// </summary>
        private static readonly ShaderTypes[] ShaderTypesToAdd = new ShaderTypes[]
        {
            ShaderTypes.SpecDeclarationGenericShader,
            ShaderTypes.SpecDeclarationTargetShader,
            ShaderTypes.ParticleDeclarationShader,
            ShaderTypes.ParticleDeclarationVertexShader,
            ShaderTypes.ParticleDeclarationFragmentShader,
            ShaderTypes.StreamOutDeclarationVertexShader,
            ShaderTypes.ParticleVertexShader,
            ShaderTypes.ParticleFragmentShader,
            ShaderTypes.StreamOutVertexShader,
            ShaderTypes.GeneralVertexShader,
            ShaderTypes.GeneralFragmentShader,
            ShaderTypes.GeneralComputeShader,
        };

        /// <summary>The emitter sets to output the containing textures.</summary>
        private EmitterSetData[] emitterSets;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="emitterSets">The emitterSet</param>
        /// <param name="eftCombinerShaderPaths">
        /// The file paths of the shader generated by the effect combiner.
        /// </param>
        public ShaderArrayWriter(
            IEnumerable<EmitterSetData> emitterSets,
            IEnumerable<string> eftCombinerShaderPaths)
        {
            this.emitterSets = emitterSets.ToArray();

            // Prepare the shader array data model.
            // We need the data model to select binary header later.
            var shaderArray = new ShaderArrayData();

            // Add shader source codes.
            foreach (ShaderTypes shaderType in ShaderTypesToAdd)
            {
                string shaderCode = CustomShaderUserDataManager.GetShaderSourceCode(shaderType);
                if (string.IsNullOrEmpty(shaderCode) == false)
                {
                    shaderArray.AddShader(shaderType, shaderCode);
                }
            }

            // 使用中のカスタムシェーダインデックスを列挙
            var customShaderIndices = (from eset in this.emitterSets
                                       from idx in eset.CustomShaderIndices
                                       where idx >= 0
                                       select idx).Distinct();

            // カスタムシェーダパスを追加
            foreach (var index in customShaderIndices)
            {
                var def = CustomShaderUserDataManager.GetCustomShaderDefinition(index);
                if (def == null)
                {
                    continue;
                }

                string vertexShaderCode = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.CustomVertexShader,
                    index);

                if (string.IsNullOrEmpty(vertexShaderCode) == false)
                {
                    shaderArray.AddCustomShader(
                        ShaderTypes.CustomVertexShader,
                        def.Id,
                        vertexShaderCode);
                }

                string fragmentShaderCode = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.CustomFragmentShader,
                    index);

                if (string.IsNullOrEmpty(fragmentShaderCode) == false)
                {
                    shaderArray.AddCustomShader(
                        ShaderTypes.CustomFragmentShader,
                        def.Id,
                        fragmentShaderCode);
                }
            }

            // 使用中のリサーブドシェーダIDを列挙
            var reservedShaderIds = (from eset in this.emitterSets
                                     from emitter in eset.AllActiveChildEmitters
                                     where emitter.ActiveReservedShader != null &&
                                           emitter.ActiveReservedShader.PageData != null &&
                                           emitter.ActiveReservedShader.PageData.ContentsData != null
                                     select ReservedShaderUserDataManager.FindReservedShaderIndexByUserData(emitter.ActiveReservedShader.PageData.ContentsData)).Distinct();

            // リサーブドシェーダパスを追加
            foreach (var index in reservedShaderIds)
            {
                var def = ReservedShaderUserDataManager.GetReservedShaderDefinition(index);
                if (def == null)
                {
                    continue;
                }

                string vertexShaderCode = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.ReservedVertexShader,
                    index);

                if (string.IsNullOrEmpty(vertexShaderCode) == false)
                {
                    shaderArray.AddCustomShader(
                        ShaderTypes.ReservedVertexShader,
                        def.Id,
                        vertexShaderCode);
                }

                string fragmentShaderCode = CustomShaderUserDataManager.GetShaderSourceCode(
                    ShaderTypes.ReservedFragmentShader,
                    index);

                if (string.IsNullOrEmpty(fragmentShaderCode) == false)
                {
                    shaderArray.AddCustomShader(
                        ShaderTypes.ReservedFragmentShader,
                        def.Id,
                        fragmentShaderCode);
                }
            }

            // Add the effect combiner generated shader paths.
            foreach (string path in eftCombinerShaderPaths)
            {
                var combiner = EffectCombinerCommunicationManager.CommunicationBridge;
                if (combiner != null)
                {
                    string shaderCode = combiner.GetShaderSourceCode(path);
                    if (string.IsNullOrEmpty(shaderCode) == false)
                    {
                        shaderArray.AddShader(ShaderTypes.EffectCombinerShader, shaderCode);
                    }
                }
            }

            this.DataModel = shaderArray;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the shader resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var shaderArray = this.DataModel as ShaderArrayData;
            if (shaderArray == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            // Write all the texture resource data to the stream.
            foreach (ShaderResourceData data in shaderArray.ShaderResources)
            {
                // Create a writer for the data.
                var writer = new ShaderResourceWriter(data);

                // Write the data to the stream.
                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            // Save the size of the binary data.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
