﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.BinaryHeaders.Helpers;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.SpecDefinitions;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

using EffectMaker.Foundation.Collections;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Primitive
{
    /// <summary>
    /// Write texture array to binary stream.
    /// </summary>
    public class PrimitiveArrayWriter : IBinaryResourceWriter
    {
        /// <summary>The emitter sets to output the containing textures.</summary>
        private EmitterSetData[] emitterSets;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="emitterSets">The emitterSet</param>
        public PrimitiveArrayWriter(IEnumerable<EmitterSetData> emitterSets)
        {
            this.emitterSets = emitterSets.ToArray();

            // Prepare the texture array data model.
            // We need the data model to select binary header later.
            var primitiveArray = new PrimitiveArrayData();

            IEnumerable<Tuple<EmitterSetData, string, string>> primitives;

            if (SpecManager.CurrentSpec.BinaryHeader == "VFXB")
            {
                // VFXBの場合はエミッタ形状とボリュームデータで利用されているもの
                primitives = this.emitterSets.SelectMany(eset =>
                {
                    return eset.EnumerateEmitterNameAndOriginalPrimitivePaths()
                        .Select(x => new Tuple<EmitterSetData, string, string>(eset, x.Key, x.Value));
                });
            }
            else
            {
                // EFTBの場合は全プリミティブを集める
                primitives = this.emitterSets.SelectMany(eset =>
                {
                    return eset.EnumerateEmitterNameAndOriginalPrimitivePaths()
                        .Union(eset.EnumerateEmitterNameAndPrimitivePaths())
                        .Select(x => new Tuple<EmitterSetData, string, string>(eset, x.Key, x.Value));
                });
            }

            // Check primitive file existence.
            foreach (Tuple<EmitterSetData, string, string> data in primitives)
            {
                if (File.Exists(data.Item3) == false)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "Primitive \"{0} / {1} / {2}\" is missing. (eset file at {3})",
                        data.Item1.Name,
                        data.Item2,
                        data.Item3,
                        data.Item1.FilePath);
                }
            }

            // Convert all the primitive paths to lower case and discard repeated ones.
            var primitivePaths = primitives.Select(data => data.Item3.ToLower()).Distinct();

            // Add all the distinct texture paths to the texture array data.
            this.PrimitiveCount = 0;
            foreach (string path in primitivePaths)
            {
                ulong guid = Manager.PrimitiveManager.Instance.GetGuid(path);
                if (Manager.PrimitiveManager.Instance.IsLocatedOnAnotherBinaryAsOriginalPrimitive(guid))
                {
                    // 別バイナリにあるプリミティブはスキップ
                    continue;
                }

                primitiveArray.AddPrimitivePath(path);
                ++this.PrimitiveCount;
            }

            //// トリミングプリミティブを追加

            // First collect all the primitive paths in the emitter sets.
            var trimmingTextures = this.emitterSets.SelectMany(eset =>
                {
                    return eset.EnumerateEmitterNameAndTrimmingTexturePaths()
                        .Select(x => new Tuple<EmitterSetData, string, string>(eset, x.Key, x.Value));
                });

            // Check primitive file existence.
            foreach (Tuple<EmitterSetData, string, string> data in trimmingTextures)
            {
                if (File.Exists(data.Item3) == false)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "Trimming texture \"{0} / {1} / {2}\" is missing. (eset file at {3})",
                        data.Item1.Name,
                        data.Item2,
                        data.Item3,
                        data.Item1.FilePath);
                }
            }

            // Convert all the trimming texture paths to lower case and discard repeated ones.
            var trimmingTexturePaths = trimmingTextures.Select(data => data.Item3.ToLower()).Distinct();

            // Add all the distinct texture paths to the texture array data.
            foreach (string path in trimmingTexturePaths)
            {
                ulong guid = Manager.PrimitiveManager.Instance.GetGuid(path);
                if (Manager.PrimitiveManager.Instance.IsLocatedOnAnotherBinaryAsOriginalPrimitive(guid))
                {
                    // 別バイナリにあるプリミティブはスキップ
                    continue;
                }

                primitiveArray.AddTrimmingTexturePath(path);
            }

            this.DataModel = primitiveArray;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Get the number of primitive that is to be written.
        /// </summary>
        public int PrimitiveCount { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var primitiveArray = this.DataModel as PrimitiveArrayData;
            if (primitiveArray == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            // Write all the primitive resource data to the stream.
            foreach (PrimitiveResourceData data in primitiveArray.PrimitiveResources)
            {
                // Create a writer for the data.
                var writer = new PrimitiveResourceWriter(data);

                // Write the data to the stream.
                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            // Write all the trimming primitive resource data to the stream.
            foreach (TrimmingPrimitiveResourceData data in primitiveArray.TrimmingPrimitiveResources)
            {
                // Create a writer for the data.
                var writer = new TrimmingPrimitiveResourceWriter(data);

                // Write the data to the stream.
                if (writer.Write(context) == false)
                {
                    return false;
                }
            }

            // Save the size of the binary data.
            this.Size = stream.Position - this.Position;

            // サイズがヘッダ分しか無い場合は、バイナリ本体がないので、オフセットを無効値にする
            if (this.Size == BinaryStructHeader.Size)
            {
                this.Offset = 0xFFFFFFFF - BinaryStructHeader.Size;
            }

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
