﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Model;
using EffectMaker.Foundation.Model.Types;
using G3dIfLib;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Primitive
{
    /// <summary>
    /// G3D版のプリミティブアレイライタです。
    /// </summary>
    public class G3dPrimitiveArrayWriter : IBinaryResourceWriter
    {
        /// <summary>The emitter sets to output the containing textures.</summary>
        private EmitterSetData[] emitterSets;

        /// <summary>
        /// プリミティブバイナリとして出力するbfresファイルのパス
        /// </summary>
        private string outputPath;

        /// <summary>
        /// プリミティブバイナリをエフェクトバイナリとは別ファイルに出力する場合はtrue
        /// エフェクトバイナリに含める場合はfalse
        /// </summary>
        private bool outputBinaryFiles;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="emitterSets">エミッタセットリスト</param>
        /// <param name="outputFilePath">出力バイナリファイルパス</param>
        public G3dPrimitiveArrayWriter(IEnumerable<EmitterSetData> emitterSets,
                                       string outputFilePath)
        {
            this.emitterSets = emitterSets.ToArray();

            // Prepare the texture array data model.
            // We need the data model to select binary header later.
            var g3dPrimitiveArray = new G3dPrimitiveArrayData();

            // First collect all the primitive paths in the emitter sets.
            var primitives = this.emitterSets.SelectMany(eset =>
            {
                return eset.EnumerateEmitterNameAndPrimitivePaths()
                    .Select(x => new Tuple<EmitterSetData, string, string>(eset, x.Key, x.Value));
            });

            // Check primitive file existence.
            foreach (Tuple<EmitterSetData, string, string> data in primitives)
            {
                if (File.Exists(data.Item3) == false)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "Primitive \"{0} / {1} / {2}\" is missing. (eset file at {3})",
                        data.Item1.Name,
                        data.Item2,
                        data.Item3,
                        data.Item1.FilePath);
                }
            }

            Version latestVersion = G3dIfUtility.GetLatestFileVersion();
            if (latestVersion == null)
            {
                Logger.Log(LogLevels.Error, "Failed to check G3D XSD folder");
            }

            // Convert all the primitive paths to lower case and discard repeated ones.
            var primitivePaths = primitives.Select(data => data.Item3.ToLower()).Distinct();

            // Add all the distinct texture paths to the texture array data.
            this.PrimitiveCount = 0;
            foreach (string path in primitivePaths)
            {
                ulong guid = Manager.PrimitiveManager.Instance.GetGuid(path);
                if (Manager.PrimitiveManager.Instance.IsLocatedOnAnotherBinaryAsG3dPrimitive(guid))
                {
                    // 別バイナリにあるプリミティブはスキップ
                    continue;
                }

                if (G3dIfUtility.CheckG3DFileVersion(path, latestVersion) == true)
                {
                    g3dPrimitiveArray.AddPrimitivePath(path);
                }
                else
                {
                    Logger.Log(LogLevels.Error, "Invalid file version {0} at G3dPrimitiveArrayWriter", path);
                }

                ++this.PrimitiveCount;
            }

            this.DataModel = g3dPrimitiveArray;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;

            // ファイルパスが渡された場合は、バイナリを別ファイルで出力するとみなす
            if (string.IsNullOrEmpty(outputFilePath) == false)
            {
                this.outputPath = outputFilePath;
                this.outputBinaryFiles = true;
            }
            else
            {
                this.outputPath = null;
                this.outputBinaryFiles = false;
            }
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Get the number of primitive that is to be written.
        /// </summary>
        public int PrimitiveCount { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }
            var g3dPrimitiveArray = this.DataModel as G3dPrimitiveArrayData;
            if (g3dPrimitiveArray == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            MemoryStream memoryStream = new MemoryStream();

            if (g3dPrimitiveArray.PrimitivePaths.Any())
            {
                // コンバートを実行
                byte[] bfresData = GfxPrimitiveBinaryConverter.Instance.Convert(this.emitterSets[0].Guid,
                                                                                g3dPrimitiveArray.PrimitivePaths,
                                                                                Options.OptionStore.RuntimeOptions.IsCommandLineMode);

                if (bfresData == null)
                {
                    return false;
                }

                // プリミティブバイナリを出力
                if (this.outputBinaryFiles)
                {
                    using (var fs = new FileStream(this.outputPath, FileMode.Create))
                    {
                        fs.Write(bfresData, 0, bfresData.Length);
                    }
                }

                // bfresバイナリからアラインサイズを取得
                // nn::util::BinaryFileHeaderのアラインサイズを取り出す
                const int alignPos = 14;
                long alignment = 1;

                // 巨大なメモリ領域を取得してしまわないように64MB程度を上限とする
                if (bfresData.Length > alignPos && bfresData[alignPos] <= 26)
                {
                    alignment = 1 << bfresData[alignPos];
                    context.UpdateAlignmentSize((uint)alignment);
                }
                else
                {
                    Logger.Log("Console", LogLevels.Error, "G3d primitive alignment size is too large.");
                    return false;
                }

                // プリミティブテーブルを書き込む
                var tableWriter = new G3dPrimitiveTableWriter(g3dPrimitiveArray);
                if (tableWriter.Write(context) == false)
                {
                    return false;
                }

                // ファイルに出力する場合はエフェクトバイナリへの書き込みをスキップする
                if (this.outputBinaryFiles == false)
                {
                    // compute alignment
                    long paddingSize = stream.Position%alignment;
                    if (paddingSize > 0)
                    {
                        paddingSize = alignment - paddingSize;
                    }

                    // Write alignment with zeroed byte array.
                    stream.Write(
                        Enumerable.Repeat((byte)0, (int)paddingSize).ToArray(),
                        0,
                        (int)paddingSize);

                    // BFRES本体のオフセットを再計算
                    this.Offset = stream.Position - this.Position - BinaryStructHeader.Size;

                    // bfresバイナリを書き込む
                    stream.Write(bfresData, 0, bfresData.Length);
                }
                else
                {
                    // エフェクトバイナリに何も追加しないので、オフセットを無効値である0xFFFFFFFFにする
                    this.Offset = 0xFFFFFFFF - BinaryStructHeader.Size;
                }
            }
            else
            {
                // 何もバイナリがない場合は、オフセットを無効値である0xFFFFFFFFにする
                this.Offset = 0xFFFFFFFF - BinaryStructHeader.Size;
            }

            // Save the size of the binary data.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
