﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.DataModel;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

namespace EffectMaker.BusinessLogic.BinaryHeaders.Helpers
{
    /// <summary>
    /// Binary header helper for shader resource binary data structures.
    /// </summary>
    public class ShaderResourceBinaryHeaderHelper : IBinaryHeaderHelper
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public ShaderResourceBinaryHeaderHelper()
        {
        }

        /// <summary>
        /// Get the flag indicating whether should set the next pointer
        /// to the end of the binary data.
        /// </summary>
        public bool ShouldSetNextToEnd
        {
            get { return false; }
        }

        /// <summary>
        /// Get the alignment inserts before the next element.
        /// </summary>
        public uint AlignmentToNext
        {
            get { return 0; }
        }

        /// <summary>
        /// Check if the given data model is accepted for this binary header helper.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>True if accepted.</returns>
        public bool IsAccepted(DataModelBase dataModel)
        {
            if (dataModel is ShaderResourceData)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Get tag string for the binary header.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The tag string.</returns>
        public string GetTag(DataModelBase dataModel)
        {
            var shader = dataModel as ShaderResourceData;
            if (shader == null)
            {
                throw new ArgumentException("Unrecognized data model.");
            }

            switch (shader.ShaderType)
            {
                case ShaderTypes.SpecDeclarationGenericShader:
                    return "SHDG";

                case ShaderTypes.SpecDeclarationTargetShader:
                    return "SHDT";

                case ShaderTypes.ParticleDeclarationShader:
                    return "SHCD";

                case ShaderTypes.ParticleDeclarationVertexShader:
                    return "SHDV";

                case ShaderTypes.ParticleDeclarationFragmentShader:
                    return "SHDF";

                case ShaderTypes.StreamOutDeclarationVertexShader:
                    return "SHDC";

                case ShaderTypes.ParticleVertexShader:
                    return "SHPV";

                case ShaderTypes.ParticleFragmentShader:
                    return "SHPF";

                case ShaderTypes.StreamOutVertexShader:
                    return "SHST";

                case ShaderTypes.GeneralVertexShader:
                    return "CSVG";

                case ShaderTypes.GeneralFragmentShader:
                    return "CSFG";

                case ShaderTypes.GeneralComputeShader:
                    return "CSCG";

                case ShaderTypes.EffectCombinerShader:
                    return "SHEC";

                case ShaderTypes.CustomVertexShader:
                {
                    var customShader = (CustomShaderResourceData)shader;
                    return "CSV" + (customShader.ShaderID - 1);
                }

                case ShaderTypes.CustomFragmentShader:
                {
                    var customShader = (CustomShaderResourceData)shader;
                    return "CSF" + (customShader.ShaderID - 1);
                }

                case ShaderTypes.ReservedVertexShader:
                {
                    var reservedShader = (CustomShaderResourceData)shader;
                    return "EPV" + (reservedShader.ShaderID - 1);
                }

                case ShaderTypes.ReservedFragmentShader:
                {
                    var reservedShader = (CustomShaderResourceData)shader;
                    return "EPF" + (reservedShader.ShaderID - 1);
                }
            }

            throw new ArgumentException("Unrecognized data model.");
        }

        /// <summary>
        /// Getting the next of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The next data model or null if the given is the last.</returns>
        public DataModelBase GetNext(DataModelBase current)
        {
            var data = current as ShaderResourceData;
            if (data != null)
            {
                var shaderArray = data.Parent as ShaderArrayData;
                if (shaderArray != null)
                {
                    bool found = false;
                    foreach (ShaderResourceData currData in shaderArray.ShaderResources)
                    {
                        if (found == false &&
                            object.ReferenceEquals(currData, data) == true)
                        {
                            // The current data is found, set up the flag.
                            found = true;
                        }
                        else if (found == true)
                        {
                            // This is the next one, return it.
                            return currData;
                        }
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// Getting the first child of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first child data model or null if the child does not exist.</returns>
        public DataModelBase GetChild(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the first data model of the given data model's sub tree.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first data model of the sub tree or null there is no sub tree.</returns>
        public DataModelBase GetSub(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the offset from the binary header to the binary data.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <param name="headerPos">The position of the binary header.</param>
        /// <returns>The offset from the binary header to the binary data.</returns>
        public uint GetOffset(DataModelBase current, uint headerPos)
        {
            return BinaryStructHeader.Size;
        }
    }
}
