﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;

namespace EffectMaker.BusinessLogic.BinaryHeaders.Helpers
{
    /// <summary>
    /// Binary header helper for primitive resource binary data structures.
    /// </summary>
    public class PrimitiveResourceBinaryHeaderHelper : IBinaryHeaderHelper
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public PrimitiveResourceBinaryHeaderHelper()
        {
        }

        /// <summary>
        /// Get the flag indicating whether should set the next pointer
        /// to the end of the binary data.
        /// </summary>
        public bool ShouldSetNextToEnd
        {
            get { return false; }
        }

        /// <summary>
        /// Get the alignment inserts before the next element.
        /// </summary>
        public uint AlignmentToNext
        {
            get { return 0; }
        }

        /// <summary>
        /// Check if the given data model is accepted for this binary header helper.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>True if accepted.</returns>
        public bool IsAccepted(DataModelBase dataModel)
        {
            if (dataModel is PrimitiveResourceData)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Get tag string for the binary header.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The tag string.</returns>
        public string GetTag(DataModelBase dataModel)
        {
            return "PRIM";
        }

        /// <summary>
        /// Getting the next of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The next data model or null if the given is the last.</returns>
        public DataModelBase GetNext(DataModelBase current)
        {
            var data = current as PrimitiveResourceData;
            if (data != null)
            {
                var primitiveArray = data.Parent as PrimitiveArrayData;
                if (primitiveArray != null)
                {
                    // PrimitiveResourcesから自分のデータを探して、その次にあるデータを返す
                    bool found = false;
                    foreach (PrimitiveResourceData currData in primitiveArray.PrimitiveResources)
                    {
                        if (found == false &&
                            object.ReferenceEquals(currData, data) == true)
                        {
                            // The current data is found, set up the flag.
                            found = true;
                        }
                        else if (found == true)
                        {
                            // This is the next one, return it.
                            return currData;
                        }
                    }

                    // PrimitiveResourcesに次のデータがなかったとき、TrimingPrimitiveResourcesの先頭データを返す
                    if (found == true)
                    {
                        var enumerator = primitiveArray.TrimmingPrimitiveResources.GetEnumerator();
                        if (enumerator.MoveNext() == true)
                        {
                            return enumerator.Current;
                        }
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// Getting the first child of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first child data model or null if the child does not exist.</returns>
        public DataModelBase GetChild(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the first data model of the given data model's sub tree.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first data model of the sub tree or null there is no sub tree.</returns>
        public DataModelBase GetSub(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the offset from the binary header to the binary data.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <param name="headerPos">The position of the binary header.</param>
        /// <returns>The offset from the binary header to the binary data.</returns>
        public uint GetOffset(DataModelBase current, uint headerPos)
        {
            return BinaryStructHeader.Size;
        }
    }
}
