﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Text.RegularExpressions;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Utility for regular expression operations.
    /// </summary>
    public static class RegexUtility
    {
        /// <summary>
        /// Match tokens in the given regular expression string.
        /// </summary>
        /// <param name="expression">The regular expression string.</param>
        /// <param name="str">The string to perform string extraction.</param>
        /// <returns>The matched string tokens.</returns>
        public static string[] ExtractStrings(string expression, string str)
        {
            var regex = new Regex(expression);

            MatchCollection matches = regex.Matches(str);

            string[] matchedTokens = new string[matches.Count];
            for (int i = 0; i < matches.Count; ++i)
            {
                Match match = matches[i];
                if (match.Success == true && match.Groups.Count >= 2)
                {
                    matchedTokens[i] = match.Groups[1].Value;
                }
                else
                {
                    matchedTokens[i] = null;
                }
            }

            return matchedTokens;
        }

        /// <summary>
        /// Indent the given string with the specified number of space characters.
        /// </summary>
        /// <param name="str">The string to indent.</param>
        /// <param name="newline">The new line characters to use.</param>
        /// <param name="indent">Indent count.</param>
        /// <returns>The indented string.</returns>
        public static string IndentString(string str, string newline, int indent)
        {
            if (str == null)
            {
                return null;
            }

            string[] lines =
                str.Split(new string[] { newline }, StringSplitOptions.None);

            for (int i = 0; i < lines.Length; ++i)
            {
                string line = lines[i];
                if (string.IsNullOrEmpty(line) == false)
                {
                    lines[i] = line.PadLeft(indent + line.Length);
                }
            }

            return string.Join(newline, (object[])lines);
        }

        /// <summary>
        /// ファイルダイアログで使用するフィルタから、拡張子を':'でつなげた文字列を返す.
        /// </summary>
        /// <param name="filteringOptions">ファイルダイアログで使用するフィルタ</param>
        /// <returns>拡張子を':'でつなげた文字列</returns>
        public static string ExtractExtensions(string filteringOptions)
        {
            // 各フィルタごとに分解する.
            var filters = filteringOptions.Split('|');

            string extensions = string.Empty;
            for (int i = 1; i < filters.Length; i += 2)
            {
                var exts = filters[i].Split(';');
                for (int j = 0; j < exts.Length; j++)
                {
                    var regex = new Regex(@"\*(?<ext>\.[^ ]*)", RegexOptions.IgnoreCase | RegexOptions.Singleline);
                    Match m = regex.Match(exts[j]);

                    // 拡張子を追加
                    extensions += m.Groups["ext"].Value;

                    // 最後の拡張子でない場合は、区切り文字である':'を追加する.
                    if (i <= filters.Length - 2 || j < exts.Length - 1)
                    {
                        extensions += ":";
                    }
                }
            }

            return extensions;
        }
    }
}
