﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Threading.Tasks;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Texture
{
    /// <summary>
    /// NW4F FTXテクスチャデータ
    /// </summary>
    public class FtxTextureData : TextureData
    {
        /// <summary>
        /// オリジナルイメージのビットマップバイト列(54バイトのヘッダあり)
        /// </summary>
        private byte[] originalBitmapStream = null;

        /// <summary>
        /// DCC プラグインのプリセット名
        /// </summary>
        public string DccPreset { get; set; }

        /// <summary>
        /// テクスチャの用途のヒント情報
        /// </summary>
        public string Hint { get; set; }

        /// <summary>
        /// RGB の誤差に対して人間の目に合わせた重み付けをするか
        /// </summary>
        public bool WeightedCompress { get; set; }

        /// <summary>
        /// The texture tile mode.
        /// </summary>
        public TextureTileModes TileMode { get; set; }

        /// <summary>
        /// GX2Surface.swizzle
        /// </summary>
        public int InitialSwizzle { get; set; }

        /// <summary>
        /// スウィズル値
        /// </summary>
        public uint Swizzle { get; set; }

        /// <summary>
        /// GX2Surface.alignment
        /// </summary>
        public uint Alignment { get; set; }

        /// <summary>
        /// GX2Surface.pitch
        /// </summary>
        public uint Pitch { get; set; }

        /// <summary>
        /// ミップマップを生成するときのフィルタ
        /// </summary>
        public string MipGenFilter { get; set; }

        /// <summary>
        /// コンポーネントセレクタ
        /// </summary>
        public ColorComponents[] CompSel { get; set; }

        /// <summary>
        /// ミップマップオフセット
        /// </summary>
        public uint[] MipmapOffset { get; set; }

        /// <summary>
        /// ネイティブイメージ
        /// </summary>
        public byte[] NativeImageStream { get; set; }

        /// <summary>
        /// コンバータの名前をテクスチャごとに設定しておく
        /// </summary>
        public string ConverterName { get; set; }

        /// <summary>
        /// テクスチャデータのメモリーサイズを取得します。
        /// </summary>
        public override int MemorySize
        {
            get
            {
                return base.MemorySize + this.NativeImageStream.Length;
            }
        }

        /// <summary>
        /// アルファチャンネルを持つかどうか取得します。
        /// </summary>
        public override bool HasAlpha
        {
            get
            {
                return base.HasAlpha || (this.CompSel[3] != ColorComponents.Item1);
            }
        }

        /// <summary>
        /// 引数で渡されたストリームにオリジナルイメージのビットマップを書き込みます。
        /// 配列テクスチャ、ミップマップには未対応です。
        /// </summary>
        /// <param name="stream">オリジナルイメージを書き込むストリーム</param>
        /// <returns>書き込めたらtrue,失敗したらfalse.</returns>
        public void WriteOriginalBitmapStream(Stream stream)
        {
            if (this.originalBitmapStream == null)
            {
                this.MakeOriginalBitmap();
            }

            // ヘッダ分ずらして出力ストリームに書き込む
            stream.Write(this.originalBitmapStream, 54, this.originalBitmapStream.Length - 54);
        }

        /// <summary>
        /// オリジナルイメージを生成します。
        /// 配列テクスチャ、ミップマップには未対応です。
        /// </summary>
        private void MakeOriginalBitmap()
        {
            Func<Bitmap, ColorMatrix, PixelFormat, Bitmap> conv = (src, matrix, format) =>
            {
                var image = new Bitmap(src.Width, src.Height, format);

                using (var ia = new ImageAttributes())
                using (var g = Graphics.FromImage(image))
                {
                    ia.SetColorMatrix(matrix);
                    g.DrawImage(
                            src,
                            new Rectangle(0, 0, src.Width, src.Height),
                            0,
                            0,
                            src.Width,
                            src.Height,
                            GraphicsUnit.Pixel,
                            ia);
                }

                return image;
            };

            Bitmap convImg;

            // RGBチャンネルが同じColorComponentを指していたら、2チャンネルテクスチャにする
            if (this.CompSel[0] == this.CompSel[1] && this.CompSel[1] == this.CompSel[2])
            {
                convImg = this.GenerateLuminanceAlphaBitMap(0, 0);
            }
            else
            {
                // 取り出したビットマップをRとBを入れ替えて変換する
                var bmp = this.GenerateBitmap(0, 0);
                var tmpComp = new[]
                {
                    this.CompSel[2], this.CompSel[1], this.CompSel[0], this.CompSel[3],
                };

                convImg = this.HasAlpha
                    ? conv(
                        bmp,
                        RenderUtility.CreateRgbaColorMatrix(tmpComp, this.PixelFormat),
                        System.Drawing.Imaging.PixelFormat.Format32bppArgb)
                    : conv(
                        bmp,
                        RenderUtility.CreateRgbColorMatrix(tmpComp, this.PixelFormat),
                        System.Drawing.Imaging.PixelFormat.Format32bppArgb);

                bmp.Dispose();
            }

            // 上下フリップして一時ストリームに書き込み、バイト列をメンバに保存する
            var ms = new MemoryStream();
            convImg.RotateFlip(RotateFlipType.RotateNoneFlipY);
            convImg.Save(ms, System.Drawing.Imaging.ImageFormat.Bmp);
            convImg.Dispose();

            this.originalBitmapStream = ms.ToArray();
        }

        /// <summary>
        /// RAチャンネルだけのビットマップを生成する.
        /// 実際は32ビットのビットマップに、RG, BAチャンネルにそれぞれ別ピクセルのデータを入れてしまう.
        /// </summary>
        /// <param name="arrayIndex">配列インデックス</param>
        /// <param name="mipLevel">ミップマップレベル</param>
        /// <returns>RAチャンネルだけに圧縮したビットマップ</returns>
        private Bitmap GenerateLuminanceAlphaBitMap(int arrayIndex, int mipLevel)
        {
            Debug.Assert(arrayIndex < this.ArraySize, "配列インデックスの指定が不正");
            Debug.Assert(mipLevel < this.MipmapCount, "ミップマップの指定が不正");

            var dstWidth = Math.Max(this.Width >> mipLevel, 1) / 2;
            var dstHeight = Math.Max(this.Height >> mipLevel, 1);

            // コンバート先のビットマップ
            Bitmap dstBitmap = new Bitmap(dstWidth, dstHeight, System.Drawing.Imaging.PixelFormat.Format32bppArgb);
            var rect = new Rectangle(0, 0, dstWidth, dstHeight);
            BitmapData dstBitmapData = dstBitmap.LockBits(rect, ImageLockMode.WriteOnly,
                System.Drawing.Imaging.PixelFormat.Format32bppArgb);

            // コンバート元のビットマップ
            Bitmap srcBitmap = this.GenerateBitmap(arrayIndex, mipLevel);
            var srcRect = new Rectangle(0, 0, this.Width, this.Height);
            BitmapData srcBitmapData = srcBitmap.LockBits(srcRect, ImageLockMode.ReadOnly,
                System.Drawing.Imaging.PixelFormat.Format32bppArgb);

            // Rチャンネルのオフセットを調べる
            int rOffset = this.GetConvertedColorComponents(this.CompSel[0]);

            // Aチャンネルのオフセットを調べる
            int aOffset = this.GetConvertedColorComponents(this.CompSel[3]);

            try
            {
                unsafe
                {
                    var dst = (byte*)dstBitmapData.Scan0.ToPointer();
                    var src = (byte*)srcBitmapData.Scan0.ToPointer();

                    Parallel.For(
                    0,
                    dstBitmapData.Height,
                    y =>
                    {
                        for (int x = 0; x < dstBitmapData.Stride; x += 2)
                        {
                            int dstPos = y * dstBitmapData.Stride + x;
                            int srcPos = y * dstBitmapData.Stride * 2 + x * 2;

                            // Rチャンネル(定数値を利用する場合はここで入れる)
                            dst[dstPos] =
                                rOffset == (uint)ColorComponents.Item0 ? (byte)0 :
                                rOffset == (uint)ColorComponents.Item1 ? (byte)255 :
                                src[srcPos + rOffset];

                            // Aチャンネル(定数値を利用する場合はここで入れる)
                            dst[dstPos + 1] =
                                aOffset == (uint)ColorComponents.Item0 ? (byte)0 :
                                aOffset == (uint)ColorComponents.Item1 ? (byte)255 :
                                src[srcPos + aOffset];
                        }

                    });
                }
            }
            finally
            {
                dstBitmap.UnlockBits(dstBitmapData);
                srcBitmap.UnlockBits(srcBitmapData);
            }

            srcBitmap.Dispose();

            return dstBitmap;
        }

        /// <summary>
        /// ColorComponentを入れ替える
        /// </summary>
        /// <param name="i">元になるColorComponent</param>
        /// <returns>RとBチャンネルを入れ替えたColorComponent</returns>
        private int GetConvertedColorComponents(ColorComponents i)
        {
            if (i == ColorComponents.Red)
            {
                return (int)ColorComponents.Blue;
            }
            else if (i == ColorComponents.Blue)
            {
                return (int)ColorComponents.Red;
            }
            else
            {
                return (int)i;
            }
        }
    }
}
