﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;

using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class to represent the transformation on an IRenderable.
    /// </summary>
    public class Transformation : ISettable, ICloneable
    {
        /// <summary>The scale to apply to the rendering.</summary>
        private PointF scale = new PointF(1.0f, 1.0f);

        /// <summary>The translation to apply to the rendering.</summary>
        private PointF translation = new PointF(0.0f, 0.0f);

        /// <summary>The transformation matrix.</summary>
        private Matrix transformationMatrix = new Matrix();

        /// <summary>Flag indicating whether the transformation matrix needs update.</summary>
        private bool isTransMatrixDirty = false;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public Transformation()
        {
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source to copy from.</param>
        public Transformation(Transformation src)
        {
            this.Set(src);
        }

        /// <summary>
        /// Get or set the scale.
        /// </summary>
        public PointF Scale
        {
            get
            {
                return this.scale;
            }

            set
            {
                this.scale = value;
                this.isTransMatrixDirty = true;
            }
        }

        /// <summary>
        /// Get or set the translation.
        /// </summary>
        public PointF Translation
        {
            get
            {
                return this.translation;
            }

            set
            {
                this.translation = value;
                this.isTransMatrixDirty = true;
            }
        }

        /// <summary>
        /// Get the transformation matrix.
        /// </summary>
        public Matrix TransformationMatrix
        {
            get
            {
                // Do we need to update the matrix?
                if (this.isTransMatrixDirty == true)
                {
                    this.transformationMatrix.Reset();
                    this.transformationMatrix.Scale(this.scale.X, this.scale.Y);
                    this.transformationMatrix.Translate(this.translation.X, this.translation.Y);

                    // Reset the flag.
                    this.isTransMatrixDirty = false;
                }

                return this.transformationMatrix;
            }
        }

        /// <summary>
        /// Clear the transformation.
        /// This will reset the transformation matrix to identity matrix.
        /// </summary>
        public void ClearTransformation()
        {
            this.translation.X = 0.0f;
            this.translation.Y = 0.0f;
            this.scale.X = 1.0f;
            this.scale.Y = 1.0f;
            this.isTransMatrixDirty = true;
        }

        /// <summary>
        /// Apply transformation.
        /// The transformation is added to the original transformation values.
        /// </summary>
        /// <param name="transformation">Translation to apply.</param>
        public void ApplyTransformation(Transformation transformation)
        {
            this.translation.X += transformation.translation.X;
            this.translation.Y += transformation.translation.Y;
            this.scale.X *= transformation.scale.X;
            this.scale.Y *= transformation.scale.Y;

            this.isTransMatrixDirty = true;
        }

        /// <summary>
        /// Apply transformation.
        /// The transformation is added to the original transformation values.
        /// </summary>
        /// <param name="translateX">Translation on X axis.</param>
        /// <param name="translateY">Translation on Y axis.</param>
        /// <param name="scaleX">Scale on X axis.</param>
        /// <param name="scaleY">Scale on Y axis.</param>
        public void ApplyTransformation(
            float translateX,
            float translateY,
            float scaleX = 1.0f,
            float scaleY = 1.0f)
        {
            this.translation.X += translateX;
            this.translation.Y += translateY;
            this.scale.X *= scaleX;
            this.scale.Y *= scaleY;

            this.isTransMatrixDirty = true;
        }

        /// <summary>
        /// Set data.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public bool Set(object src)
        {
            var srcTransformation = src as Transformation;
            if (srcTransformation == null)
            {
                return false;
            }

            this.translation = srcTransformation.translation;
            this.scale = srcTransformation.scale;

            this.transformationMatrix.Reset();
            this.transformationMatrix.Scale(this.scale.X, this.scale.Y);
            this.transformationMatrix.Translate(this.translation.X, this.translation.Y);

            this.isTransMatrixDirty = false;

            return true;
        }

        /// <summary>
        /// Clone this render context.
        /// </summary>
        /// <returns>The new render context.</returns>
        public object Clone()
        {
            return new Transformation(this);
        }
    }
}
