﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;

using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that renders a label.
    /// </summary>
    public class TextBlock : RectangleShape
    {
        /// <summary>The render color for the text.</summary>
        private Color textColor = Color.Black;

        /// <summary>The bounds that specifically assigned by user.</summary>
        private RectangleF assignedBounds = RectangleF.Empty;

        /// <summary>The text.</summary>
        private string text = string.Empty;

        /// <summary>The font for rendering the text.</summary>
        private Font font = new Font(FontFamily.GenericSansSerif, 10.0f);

        /// <summary>
        /// Default constructor.
        /// </summary>
        public TextBlock() :
            base()
        {
            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;
            this.HAlignment = HAlignment.Left;
            this.VAlignment = VAlignment.Top;
            this.AutoSize = true;
            this.Padding = new Padding(5);

            this.StringFormat = new StringFormat()
            {
                Alignment = StringAlignment.Near,
                LineAlignment = StringAlignment.Near,
                Trimming = StringTrimming.Character
            };
        }

        /// <summary>
        /// Get or set the boundary of the item.
        /// </summary>
        public override RectangleF Bounds
        {
            get
            {
                return base.Bounds;
            }

            set
            {
                base.Bounds = value;
                this.assignedBounds = value;
            }
        }

        /// <summary>
        /// Get or set the render color for the text.
        /// </summary>
        public Color TextColor
        {
            get
            {
                return this.textColor;
            }

            set
            {
                if (this.textColor.Equals(value) == false)
                {
                    this.textColor = value;
                    this.RequestRedraw();
                }
            }
        }

        /// <summary>
        /// Get or set the text.
        /// </summary>
        public string Text
        {
            get
            {
                return this.text;
            }

            set
            {
                if (this.text != value)
                {
                    this.text = value;
                    this.RequestRedraw();
                }
            }
        }

        /// <summary>
        /// Get or set the font for rendering the text.
        /// </summary>
        public Font Font
        {
            get
            {
                return this.font;
            }

            set
            {
                if (this.font.Equals(value) == false)
                {
                    if (this.font != null)
                    {
                        this.font.Dispose();
                    }

                    if (value != null)
                    {
                        this.font = value.Clone() as Font;
                    }
                    else
                    {
                        this.font = new Font(FontFamily.GenericSansSerif, 10.0f);
                    }

                    this.RequestRedraw();
                }
            }
        }

        /// <summary>
        /// Get or set the string format.
        /// </summary>
        public StringFormat StringFormat { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether this text block is auto sized by
        /// the text size.
        /// </summary>
        public bool AutoSize { get; set; }

        /// <summary>
        /// Get or set the padding for the text block.
        /// </summary>
        public Padding Padding { get; set; }

        /// <summary>
        /// Get or set the horizontal alignment for the text block.
        /// </summary>
        public HAlignment HAlignment { get; set; }

        /// <summary>
        /// Get or set the vertical alignment for the text block.
        /// </summary>
        public VAlignment VAlignment { get; set; }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            if (context.Graphics != null && this.AutoSize == true)
            {
                SizeF textSize = context.Graphics.MeasureString(this.Text, this.Font);
                textSize.Width += this.Padding.Horizontal;
                textSize.Height += this.Padding.Vertical;

                PointF pos = this.assignedBounds.Location;
                switch (this.HAlignment)
                {
                    case HAlignment.Center:
                        pos.X -= textSize.Width * 0.5f;
                        break;

                    case HAlignment.Right:
                        pos.X -= textSize.Width;
                        break;
                }

                switch (this.VAlignment)
                {
                    case VAlignment.Center:
                        pos.Y -= textSize.Height * 0.5f;
                        break;

                    case VAlignment.Bottom:
                        pos.Y -= textSize.Height;
                        break;
                }

                base.Bounds = new RectangleF(pos, textSize);
            }
            else
            {
                base.Bounds = this.assignedBounds;
            }

            base.UpdateSelf(context);
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            base.DrawForeground(g);

            // Render the text.
            if (string.IsNullOrEmpty(this.Text) == false &&
                this.TextColor.Equals(Color.Transparent) == false)
            {
                using (var brush = new SolidBrush(this.TextColor))
                {
                    RectangleF textRegion = this.ClippedBounds;
                    textRegion.X += this.Padding.Left;
                    textRegion.Y += this.Padding.Top;
                    textRegion.Width -= this.Padding.Horizontal;
                    textRegion.Height -= this.Padding.Vertical;

                    g.DrawString(this.Text, this.Font, brush, textRegion);
                }
            }
        }
    }
}
