﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;

using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that renders a line segment.
    /// </summary>
    public class ManyLineSegments : RenderableBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public ManyLineSegments()
        {
            this.Vertices = null;
            this.BorderColor = Color.Black;
            this.BorderThickness = 1.0f;
            this.PickThreshold = 3.0f;
            this.DashStyle = DashStyle.Solid;
        }

        /// <summary>
        /// Get or set the boundary of the rectangle.
        /// </summary>
        public new RectangleF Bounds
        {
            get
            {
                if (this.Vertices == null)
                {
                    return new RectangleF();
                }

                return new RectangleF(
                    this.Vertices.Min(p => p.X),
                    this.Vertices.Min(p => p.Y),
                    Math.Abs(this.Vertices.Max(p => p.X) - this.Vertices.Min(p => p.X)),
                    Math.Abs(this.Vertices.Max(p => p.Y) - this.Vertices.Min(p => p.Y)));
            }
        }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public List<PointF> Vertices { get; set; }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public List<PointF> TransformedVertices { get; protected set; }

        /// <summary>
        /// Get or set the location of an end of the curve.
        /// </summary>
        public List<PointF> ClippedVertices { get; protected set; }

        /// <summary>
        /// Get or set the flag indicating if the line segment is in the view rectangle.
        /// </summary>
        public bool IsInViewRectangle { get; protected set; }

        /// <summary>
        /// Get or set the threshold for mouse picking.
        /// </summary>
        public float PickThreshold { get; set; }

        /// <summary>
        /// Get or set the fill color of the item.
        /// </summary>
        public new Color FillColor
        {
            get { return Color.Transparent; }
        }

        /// <summary>
        /// Gets or sets the dash style.
        /// </summary>
        public DashStyle DashStyle { get; set; }

        /// <summary>
        /// Test if the given point lies on this object.
        /// </summary>
        /// <param name="p">The point.</param>
        /// <returns>True if the point picks this object.</returns>
        public override bool Pick(PointF p)
        {
            if (this.CanPick == false || this.Visible == false)
            {
                return false;
            }

            if (this.TransformedVertices == null)
            {
                this.UpdateSelf(null);
            }

            for (int i = 0; i < this.TransformedVertices.Count - 1; i += 2)
            {
                // Compute the distance from the point.
                float distance = MathUtility.ComputePointLineDistance(
                    p,
                    this.TransformedVertices[i],
                    this.TransformedVertices[i + 1]);
                if (distance <= this.PickThreshold)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// Render the item.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>False when the item is clipped and need not to be rendered.</returns>
        public override bool Draw(Graphics g)
        {
            if (this.Visible == false)
            {
                return false;
            }

            if (this.BorderThickness <= 0.0f)
            {
                return false;
            }

            if (this.TransformedVertices == null)
            {
                // The line segment's length is zero, don't render it.
                return false;
            }

            if (this.IsInViewRectangle == false)
            {
                // The line segment does not lie in the view rectangle, don't render it.
                return false;
            }

            if (this.IsUpdateAndRenderingSuspended == false)
            {
                this.SuspendUpdateAndRendering();

                this.DrawBackground(g);
                this.DrawForeground(g);
                this.DrawBorder(g);
                this.DrawChildren(g);

                this.ResumeUpdateAndRendering();

                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            base.UpdateSelf(context);

            PointF[] points = this.Vertices.ToArray();

            // Transform the vertices with the transformation matrix.
            context.TransformationMatrix.TransformPoints(points);

            this.TransformedVertices = points.ToList();

            this.TransformedBounds =
                new RectangleF(
                    this.TransformedVertices.Min(p => p.X),
                    this.TransformedVertices.Min(p => p.Y),
                    Math.Abs(this.TransformedVertices.Max(p => p.X) - this.TransformedVertices.Min(p => p.X)),
                    Math.Abs(this.TransformedVertices.Max(p => p.Y) - this.TransformedVertices.Min(p => p.Y)));

            // Do clipping with the view rectangle.
            if (this.ClippedVertices == null)
            {
                this.ClippedVertices = new List<PointF>();
            }

            this.ClippedVertices.Clear();
            foreach (var p in this.TransformedVertices)
            {
                this.ClippedVertices.Add(new PointF());
            }

            for (int i = 0; i < this.TransformedVertices.Count - 1; i += 2)
            {
                PointF p1 = this.TransformedVertices[i];
                PointF p2 = this.TransformedVertices[i + 1];
                if (context.ViewRectangle.IsEmpty == false)
                {
                    this.IsInViewRectangle = RenderUtility.ClipLineSegment(
                        context.ViewRectangle,
                        this.TransformedVertices[i],
                        this.TransformedVertices[i + 1],
                        out p1,
                        out p2);
                }
                else
                {
                    this.IsInViewRectangle = false;
                }

                this.ClippedVertices[i] = p1;
                this.ClippedVertices[i + 1] = p2;
            }

            this.ClippedBounds =
                new RectangleF(
                    this.ClippedVertices.Min(p => p.X),
                    this.ClippedVertices.Min(p => p.Y),
                    Math.Abs(this.ClippedVertices.Max(p => p.X) - this.ClippedVertices.Min(p => p.X)),
                    Math.Abs(this.ClippedVertices.Max(p => p.Y) - this.ClippedVertices.Min(p => p.Y)));

            this.IsInViewRectangle = true;
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBorder(Graphics g)
        {
            // Draw the line segment.
            using (Pen pen = new Pen(this.BorderColor, this.BorderThickness))
            {
                pen.DashStyle = this.DashStyle;
                if (this.DashStyle == DashStyle.Dash)
                {
                    pen.DashPattern = new float[] { 5.0f, 5.0f };
                }

                for (int i = 0; i < this.ClippedVertices.Count; i += 2)
                {
                    g.DrawLine(pen, this.ClippedVertices[i], this.ClippedVertices[i + 1]);
                }
            }
        }
    }
}
