﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;

using EffectMaker.Foundation.Render.Layout;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that renders a label.
    /// </summary>
    public class ImageRenderable : RectangleShape
    {
        /// <summary>The render color for the text.</summary>
        private Image image;

        /// <summary>The bounds that specifically assigned by user.</summary>
        private RectangleF assignedBounds = RectangleF.Empty;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public ImageRenderable() :
            base()
        {
            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;
            this.HAlignment = HAlignment.Left;
            this.VAlignment = VAlignment.Top;
            this.AutoSize = true;
            this.DotByDot = true;
            this.Padding = new Padding(0);
        }

        /// <summary>
        /// Get or set the boundary of the item.
        /// </summary>
        public override RectangleF Bounds
        {
            get
            {
                return base.Bounds;
            }

            set
            {
                base.Bounds = value;
                this.assignedBounds = value;
            }
        }

        /// <summary>
        /// Get or set the image.
        /// </summary>
        public Image Image
        {
            get
            {
                return this.image;
            }

            set
            {
                if (this.image != null)
                {
                    if (this.image.Equals(value))
                    {
                        return;
                    }
                }

                this.image = value;
                this.RequestRedraw();
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether this text block is auto sized by
        /// the text size.
        /// </summary>
        public bool AutoSize { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether dot by dot.
        /// </summary>
        public bool DotByDot { get; set; }

        /// <summary>
        /// Get or set the padding for the text block.
        /// </summary>
        public Padding Padding { get; set; }

        /// <summary>
        /// Get or set the horizontal alignment for the text block.
        /// </summary>
        public HAlignment HAlignment { get; set; }

        /// <summary>
        /// Get or set the vertical alignment for the text block.
        /// </summary>
        public VAlignment VAlignment { get; set; }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateSelf(RenderContext context)
        {
            if (context.Graphics != null && this.AutoSize == true)
            {
                SizeF imageSize = this.Image.Size;
                imageSize.Width += this.Padding.Horizontal;
                imageSize.Height += this.Padding.Vertical;

                PointF pos = this.assignedBounds.Location;
                switch (this.HAlignment)
                {
                    case HAlignment.Center:
                        pos.X -= imageSize.Width * 0.5f;
                        break;

                    case HAlignment.Right:
                        pos.X -= imageSize.Width;
                        break;
                }

                switch (this.VAlignment)
                {
                    case VAlignment.Center:
                        pos.Y -= imageSize.Height * 0.5f;
                        break;

                    case VAlignment.Bottom:
                        pos.Y -= imageSize.Height;
                        break;
                }

                base.Bounds = new RectangleF(pos, imageSize);
            }
            else
            {
                base.Bounds = this.assignedBounds;
            }

            base.UpdateSelf(context);
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            base.DrawForeground(g);

            // Render the image.
            if (this.Image != null)
            {
                // クリッピングを行わないようにするためTransformedBoundsに変更
                RectangleF imageRegion = TransformedBounds;

                imageRegion.X += this.Padding.Left;
                imageRegion.Y += this.Padding.Top;
                imageRegion.Width -= this.Padding.Horizontal;
                imageRegion.Height -= this.Padding.Vertical;

                if (this.DotByDot)
                {
                    g.DrawImageUnscaled(
                        this.Image,
                        new Rectangle(
                            (int)imageRegion.X,
                            (int)imageRegion.Y,
                            (int)imageRegion.Width,
                            (int)imageRegion.Height));
                }
                else
                {
                    g.DrawImage(this.Image, imageRegion);
                }
            }
        }
    }
}
