﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Class that renders a circle.
    /// </summary>
    public class CircleShape : RenderableBase
    {
        /// <summary>The center of the circle.</summary>
        private PointF center = PointF.Empty;

        /// <summary>The horizontal and vertical radius.</summary>
        private SizeF radius = SizeF.Empty;

        /// <summary>
        /// Constructor.
        /// </summary>
        public CircleShape()
        {
            this.BorderColor = Color.Black;
            this.BorderThickness = 1.0f;
            this.FillColor = Color.White;
        }

        /// <summary>
        /// Get or set the center of the circle.
        /// </summary>
        public PointF Center
        {
            get
            {
                return this.center;
            }

            set
            {
                if (this.center.X != value.X ||
                    this.center.Y != value.Y)
                {
                    this.center = value;
                    this.Bounds = new RectangleF(
                        this.center.X - this.radius.Width,
                        this.center.Y - this.radius.Height,
                        this.radius.Width * 2.0f,
                        this.radius.Height * 2.0f);
                }
            }
        }

        /// <summary>
        /// Get or set the horizontal and vertical radius of the circle.
        /// </summary>
        public SizeF Radius
        {
            get
            {
                return this.radius;
            }

            set
            {
                if (this.radius.Width != value.Width ||
                    this.radius.Height != value.Height)
                {
                    this.radius = value;
                    this.Bounds = new RectangleF(
                        this.center.X - this.radius.Width,
                        this.center.Y - this.radius.Height,
                        (this.radius.Width * 2.0f) + 1.0f,
                        (this.radius.Height * 2.0f) + 1.0f);
                }
            }
        }

        /// <summary>
        /// Test if the given point lies on this object.
        /// </summary>
        /// <param name="p">The point.</param>
        /// <returns>True if the point picks this object.</returns>
        public override bool Pick(PointF p)
        {
            if (this.CanPick == false || this.Visible == false)
            {
                return false;
            }

            if (this.ClippedBounds.Contains(p) == false)
            {
                return false;
            }

            if (this.TransformedBounds.Width == this.TransformedBounds.Height)
            {
                // This is a circle.
                float radius = this.TransformedBounds.Width * 0.5f;
                PointF center = this.TransformedBounds.GetCenter();

                float diffX = p.X - center.X;
                float diffY = p.Y - center.Y;

                if ((diffX * diffX) + (diffY * diffY) <= (radius * radius))
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
            else
            {
                // This is a ellipse.
                float radiusX = this.TransformedBounds.Width * 0.5f;
                float radiusY = this.TransformedBounds.Height * 0.5f;
                PointF center = this.TransformedBounds.GetCenter();

                float diffX = p.X - center.X;
                float diffY = p.Y - center.Y;

                float radiusX2 = radiusX * radiusX;
                float radiusY2 = radiusY * radiusY;

                if (((diffX * diffX) / radiusX2) + ((diffY * diffY) / radiusY2) <= 1.0f)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            // Fill the circle.
            if (this.FillColor.Equals(Color.Transparent) == false)
            {
                using (var brush = new SolidBrush(this.FillColor))
                {
                    // TODO : View rectangle clipping!
                    g.FillEllipse(brush, this.TransformedBounds);
                }
            }
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBorder(Graphics g)
        {
            // Draw the border.
            if (this.BorderThickness > 0.0f &&
                this.BorderColor.Equals(Color.Transparent) == false)
            {
                using (var pen = new Pen(this.BorderColor, this.BorderThickness))
                {
                    // TODO : View rectangle clipping!
                    g.DrawEllipse(pen, this.TransformedBounds);
                }
            }
        }
    }
}
