﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Globalization;

using EffectMaker.Foundation.Core;

namespace EffectMaker.Foundation.Primitives
{
    /// <summary>
    /// The base primitive class with floating point elements.
    /// </summary>
    public abstract class Primitivef : PrimitiveBase<float>
    {
        /// <summary>
        /// Default constructor.
        /// </summary>
        public Primitivef()
        {
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source to copy from.</param>
        public Primitivef(Primitivef src)
            : base(src)
        {
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source array.</param>
        public Primitivef(float[] src)
            : base(src)
        {
        }

        /// <summary>
        /// Get the type of the values the primitive contains.
        /// </summary>
        public override ValueTypes ValueType
        {
            get { return ValueTypes.Float; }
        }

        /// <summary>
        /// Set all the values of the primitive to zero.
        /// </summary>
        public override void SetZero()
        {
            for (int i = 0; i < this.Count; ++i)
            {
                this.Values[i] = 0.0f;
            }
        }

        /// <summary>
        /// Set all the values of the primitive to one.
        /// </summary>
        public override void SetOne()
        {
            for (int i = 0; i < this.Count; ++i)
            {
                this.Values[i] = 1.0f;
            }
        }

        /// <summary>
        /// Parse the given string to a value of the element type.
        /// </summary>
        /// <param name="strElement">The string to parse.</param>
        /// <param name="value">The parsed value.</param>
        /// <returns>True on success.</returns>
        protected override bool ParseElement(
                                             string strElement,
                                             out float value)
        {
            return float.TryParse(strElement,
                                  NumberStyles.Float | NumberStyles.AllowThousands | NumberStyles.AllowExponent,
                                  CultureInfo.InvariantCulture,
                                  out value);
        }
    }
}
