﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading;

namespace EffectMaker.Foundation.Log
{
    /// <summary>
    /// スレッドセーフなロガーです。
    /// </summary>
    public class ThreadSafeLogger
    {
        /// <summary>
        /// スレッド
        /// </summary>
        private Thread thread;

        /// <summary>
        /// ログハンドラのリスト
        /// </summary>
        private Dictionary<string, ILogHandler> logHandlers = new Dictionary<string, ILogHandler>();

        /// <summary>
        /// メッセージキュー
        /// </summary>
        private Queue<LogMessage> messageQueue = new Queue<LogMessage>();

        /// <summary>
        /// スレッド再ループイベント
        /// </summary>
        private AutoResetEvent invokeEvent = new AutoResetEvent(false);

        /// <summary>
        /// スレッド停止フラグ
        /// </summary>
        private bool stopFlag = false;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ThreadSafeLogger()
        {
        }

        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~ThreadSafeLogger()
        {
            Debug.Assert(this.thread == null);
            Debug.Assert(this.logHandlers.Count == 0);
        }

        /// <summary>
        /// スレッドを初期化します。
        /// </summary>
        /// <returns>処理が成功したときtrueを返します。</returns>
        public bool Initialize()
        {
            // スレッドを作成
            this.thread = new Thread(this.ThreadProcess);
            this.thread.Name = "EffectMaker.LoggerThread";

            // スレッドを開始
            this.thread.Start();

            return true;
        }

        /// <summary>
        /// スレッドを破棄します。
        /// </summary>
        public void Release()
        {
            // メッセージキューが0になるまで待機
            bool resWait = this.WaitQueue(3939);

            if (resWait == false)
            {
                Debug.Print("ThreadSafeLogger::Release: Timeout WaitQueue().");
            }

            // スレッド停止フラグをセット
            this.stopFlag = true;

            // 再ループイベントをセットしてスレッドを起こす
            this.invokeEvent.Set();

            // スレッドの停止を待つ
            this.thread.Join();

            // スレッドを破棄
            this.thread = null;
        }

        /// <summary>
        /// ログハンドラを登録します。
        /// </summary>
        /// <param name="handler">ログハンドラ</param>
        public void RegisterLogHandler(ILogHandler handler)
        {
            // メッセージキューが0になるまで待機
            bool resWait = this.WaitQueue(3939);

            if (resWait == false)
            {
                Debug.Print("ThreadSafeLogger::RegisterLogHandler: Timeout WaitQueue().");
            }

            // ログハンドラを登録する
            lock (this.logHandlers)
            {
                Debug.Assert(handler != null);
                Debug.Assert(this.logHandlers.ContainsKey(handler.LogHandlerName) == false);

                this.logHandlers.Add(handler.LogHandlerName, handler);
            }
        }

        /// <summary>
        /// ログハンドラの登録を解除します。
        /// </summary>
        /// <param name="handler">ログハンドラ</param>
        public void UnregisterLogHandler(ILogHandler handler)
        {
            // メッセージキューが0になるまで待機
            bool resWait = this.WaitQueue(3939);

            if (resWait == false)
            {
                Debug.Print("ThreadSafeLogger::UnregisterLogHandler: Timeout WaitQueue().");
            }

            // ログハンドラの登録を解除する
            lock (this.logHandlers)
            {
                Debug.Assert(handler != null);
                Debug.Assert(this.logHandlers.ContainsKey(handler.LogHandlerName) == true);

                this.logHandlers.Remove(handler.LogHandlerName);
            }
        }

        /// <summary>
        /// ログメッセージをキューに追加します。
        /// </summary>
        /// <param name="destinations">ログ出力先</param>
        /// <param name="logLevel">ログレベル</param>
        /// <param name="message">ログメッセージ</param>
        /// <param name="callStack">コールスタック</param>
        public void EnqueueMessage(string[] destinations, LogLevels logLevel, string message, StackFrame callStack)
        {
            // キューに追加するメッセージを作成
            LogMessage logMessage = new LogMessage()
            {
                Destinations = destinations,
                LogLevel = logLevel,
                Message = message,
                CallStack = callStack
            };

            // メッセージをキューに追加する
            lock (this.messageQueue)
            {
                this.messageQueue.Enqueue(logMessage);

                // メッセージの数が0から1になったとき
                // 再ループイベントをセットしてスレッドを再開させる
                if (this.messageQueue.Count == 1)
                {
                    this.invokeEvent.Set();
                }
            }
        }

        /// <summary>
        /// スレッドで実行する処理です。
        /// </summary>
        private void ThreadProcess()
        {
            // スレッドループ
            while (true)
            {
                int messageCount;
                LogMessage message = new LogMessage();

                // メッセージを取得
                lock (this.messageQueue)
                {
                    messageCount = this.messageQueue.Count;

                    if (messageCount > 0)
                    {
                        message = this.messageQueue.Dequeue();
                    }
                }

                // メッセージをログハンドラに送る
                if (messageCount > 0)
                {
                    List<ILogHandler> handlers = new List<ILogHandler>();

                    // 出力先リストで指定されたログハンドラを列挙する
                    lock (this.logHandlers)
                    {
                        if (message.Destinations != null && message.Destinations.Length > 0)
                        {
                            foreach (string destination in message.Destinations)
                            {
                                ILogHandler handler;
                                this.logHandlers.TryGetValue(destination, out handler);

                                if (handler == null)
                                {
                                    Debug.WriteLine("Logger.Log : The specified log destination '{0}' is not found or not yet registered.", destination);
                                    continue;
                                }

                                handlers.Add(handler);
                            }
                        }
                        else
                        {
                            handlers.Capacity = this.logHandlers.Count;
                            handlers.AddRange(this.logHandlers.Values);
                        }

                        // メッセージをログハンドラに送る
                        // handlersはthis.logHandlersのコピーでログハンドラのRelease後にログ出力処理が呼び出される場合がある。
                        // その場合はログハンドラで処理を無視する必要がある。
                        foreach (ILogHandler handler in handlers)
                        {
                            SynchronizationContext syncContext = handler.SynchronizationContext;

                            if (syncContext != null)
                            {
                                syncContext.Post(_ => handler.Log(message.Destinations, message.LogLevel, message.Message, message.CallStack), null);
                            }
                            else
                            {
                                handler.Log(message.Destinations, message.LogLevel, message.Message, message.CallStack);
                            }
                        }
                    }

                    // Visual Studioの出力ウィンドウにログを送る
                    #if DEBUG
                    {
                        string file = message.CallStack.GetFileName();
                        int line = message.CallStack.GetFileLineNumber();

                        string msg = string.Format("{0}({1}): {2}", file, line, message.Message);

                        Debug.Print(msg);
                    }
                    #endif
                }

                // 再ループイベントがセットされるまで待つ
                if (messageCount == 0)
                {
                    this.invokeEvent.WaitOne();

                    if (this.stopFlag)
                    {
                        break;
                    }
                }
            }

            this.stopFlag = false;
        }

        /// <summary>
        /// メッセージキューの数が0になるまで待機します。
        /// ここでは、ログが出力されたかどうかまでは確認しません。
        /// </summary>
        /// <param name="millisecondsTimeout">だいたいのタイムアウト時間</param>
        /// <returns>タイムアウトしたときはfalseを返します。</returns>
        private bool WaitQueue(int millisecondsTimeout)
        {
            const int sleepTime = 100;

            int loopCount = millisecondsTimeout / sleepTime;
            int i;

            // メッセージキューの数が0になるまで待機
            for (i = 0; i <= loopCount; ++i)
            {
                lock (this.messageQueue)
                {
                    if (this.messageQueue.Count == 0)
                    {
                        break;
                    }
                }

                if (i < loopCount)
                {
                    Thread.Sleep(sleepTime);
                }
            }

            return i <= loopCount;
        }

        /// <summary>
        /// ログメッセージです。
        /// </summary>
        private struct LogMessage
        {
            /// <summary>
            /// 出力先リスト
            /// </summary>
            public string[] Destinations;

            /// <summary>
            /// ログレベル
            /// </summary>
            public LogLevels LogLevel;

            /// <summary>
            /// ログメッセージ
            /// </summary>
            public string Message;

            /// <summary>
            /// コールスタック
            /// </summary>
            public StackFrame CallStack;
        }
    }
}
