﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Input
{
    /// <summary>
    /// A executable that executes an anonymous method.
    /// </summary>
    public class AnonymousExecutable : IExecutable
    {
        /// <summary>
        /// Backing field for the Execute method.
        /// </summary>
        private Action<object> execute;

        /// <summary>
        /// Backing field for the CanExecute method.
        /// </summary>
        private Predicate<object> canExecute;

        /// <summary>
        /// Backing field for the IsEnabled property.
        /// </summary>
        private bool isEnabled = true;

        /// <summary>
        /// Initialize the AnonymousExecutable instance.
        /// </summary>
        /// <param name="execute">The anonymous method to run on Execute method.</param>
        public AnonymousExecutable(Action execute)
        {
            if (execute == null)
            {
                throw new ArgumentNullException("execute");
            }

            this.execute = _ => execute();
        }

        /// <summary>
        /// Initialize the AnonymousExecutable instance.
        /// </summary>
        /// <param name="execute">The anonymous method to run on Execute method.</param>
        public AnonymousExecutable(Action<object> execute)
        {
            if (execute == null)
            {
                throw new ArgumentNullException("execute");
            }

            this.execute = execute;
        }

        /// <summary>
        /// Initialize the AnonymousExecutable instance.
        /// </summary>
        /// <param name="execute">The anonymous method to run on Execute method.</param>
        /// <param name="canExecute">The anonymous method to run on CanExecute method.</param>
        public AnonymousExecutable(Action<object> execute, Predicate<object> canExecute)
            : this(execute)
        {
            if (canExecute == null)
            {
                throw new ArgumentNullException("canExecute");
            }

            this.canExecute = canExecute;
        }

        /// <summary>
        /// Event raised when the executable availability changes.
        /// </summary>
        public event EventHandler CanExecuteChanged;

        /// <summary>
        /// Gets or sets whether the executable is available or not.
        /// </summary>
        public bool IsEnabled
        {
            get
            {
                return this.isEnabled;
            }

            set
            {
                if (this.isEnabled == value)
                {
                    return;
                }

                this.isEnabled = value;

                this.OnCanExecuteChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// Tells whether the executable can be executed given a custom parameter.
        /// </summary>
        /// <param name="parameter">Custom parameter.</param>
        /// <returns>Returns true if the executable can run, false otherwise.</returns>
        public bool CanExecute(object parameter)
        {
            if (this.IsEnabled == false)
            {
                return false;
            }

            if (this.canExecute != null)
            {
                return this.canExecute(parameter);
            }

            return true;
        }

        /// <summary>
        /// Execute the executable, providing it a custom parameter.
        /// </summary>
        /// <param name="parameter">A custom parameter.</param>
        public void Execute(object parameter)
        {
            this.execute(parameter);
        }

        /// <summary>
        /// Called when the CanExecute method would return a different value.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected virtual void OnCanExecuteChanged(EventArgs e)
        {
            var handler = this.CanExecuteChanged;

            if (handler != null)
            {
                handler(this, e);
            }
        }
    }
}
