﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.Remoting.Messaging;
using System.Threading;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using Microsoft.Win32;

namespace EffectMaker.Foundation.FileMonitor
{
    /// <summary>
    /// ファイルモニター
    /// </summary>
    public class FileMonitor : IDisposable
    {
        /// <summary>
        /// リロード待ち時間
        /// </summary>
        private static int reloadWaitTimeMs = -1;

        /// <summary>
        /// フルパス
        /// </summary>
        private readonly string fullPath;

        /// <summary>
        /// スロットル実行
        /// </summary>
        private readonly ThrottleExecuter throttleExecuter = new ThrottleExecuter();

        /// <summary>
        /// ファイルモニター
        /// </summary>
        private FileSystemWatcher watcher;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="path">モニターするパス</param>
        /// <param name="synchronizingObject">イベント通知同期オブジェクト</param>
        /// <param name="isDirectoryChildren">ディレクトリ内の子も監視する</param>
        public FileMonitor(string path, ISynchronizeInvoke synchronizingObject, bool isDirectoryChildren)
        {
            this.Path = path;
            this.fullPath = System.IO.Path.GetFullPath(path);

            // ファイルパスをチェック
            Debug.Assert(string.IsNullOrEmpty(this.fullPath) == false, "不正なファイルパス");

            string targetPath = this.fullPath;                                    // 監視対象のフルパス
            string targetName = System.IO.Path.GetFileName(targetPath);           // 監視対象のファイル名
            string targetLocation = System.IO.Path.GetDirectoryName(targetPath);  // 監視対象の親ディレクトリ

            // 監視対象のパスを設定
            {
                // ファイルがなければ存在する親ディレクトリを検索
                while (string.IsNullOrEmpty(targetLocation) == false && File.Exists(targetLocation) == false &&
                       Directory.Exists(targetLocation) == false)
                {
                    targetPath = targetLocation;
                    targetLocation = System.IO.Path.GetDirectoryName(targetLocation);

                    if (string.IsNullOrEmpty(targetLocation))
                    {
                        break;
                    }

                    targetName = System.IO.Path.GetFileName(targetPath);
                };

                // ロケーションをルートドライブにする
                // ルートドライブがないとき new FileSystemWatcher() で例外出ます...
                if (string.IsNullOrEmpty(targetLocation))
                {
                    targetLocation = targetPath;
                }

                // ルートドライブを直接指定したときは、ルートドライブ直下の全てのファイルを監視
                if (string.IsNullOrEmpty(targetName))
                {
                    targetName = "*.*";
                }

                Debug.Assert(string.IsNullOrEmpty(targetPath) == false, "指定パスが見つからない");
            }

            // ファイルウォッチャーを構築する
            // 気を付けることとして、親フォルダの名前を変更したときはイベントが発行されず
            // 勝手にパス名の変更を追尾してしまうという仕様がFileSystemWatcherにあります
            if (Directory.Exists(this.fullPath) && isDirectoryChildren)
            {
                this.watcher = new FileSystemWatcher
                {
                    NotifyFilter        = NotifyFilters.FileName | NotifyFilters.DirectoryName | NotifyFilters.Attributes | NotifyFilters.Size | NotifyFilters.LastWrite | NotifyFilters.CreationTime | NotifyFilters.Security,
                    Path                = this.fullPath,
                    Filter              = "*.*",
                    SynchronizingObject = synchronizingObject
                };
            }
            else
            {
                this.watcher = new FileSystemWatcher
                {
                    NotifyFilter        = NotifyFilters.FileName | NotifyFilters.DirectoryName | NotifyFilters.Attributes | NotifyFilters.Size | NotifyFilters.LastWrite | NotifyFilters.CreationTime | NotifyFilters.Security,
                    Path                = targetLocation,
                    Filter              = targetName,
                    SynchronizingObject = synchronizingObject
                };
            }

            this.watcher.Changed += this.Changed;
            this.watcher.Created += this.Changed;
            this.watcher.Deleted += this.Changed;
            this.watcher.Renamed += this.Changed;

            this.Start();
        }

        /// <summary>
        /// 指定した Path のファイルまたはディレクトリが変更されたときに発生します。
        /// </summary>
        public event FileSystemEventHandler FileChanged;

        /// <summary>
        /// ファイルもしくはディレクトリのパス
        /// </summary>
        public string Path { get; private set; }

        /// <summary>
        /// リロード待ち時間
        /// </summary>
        private static int ReloadReloadWaitTimeMs
        {
            get
            {
                if (reloadWaitTimeMs == -1)
                {
                    var regValue = "5000";

                    // レジストリの取得
                    try
                    {
                        // 操作するレジストリ・キーの名前
                        const string KeyName = @"Control Panel\Desktop";

                        // 取得処理を行う対象となるレジストリの値の名前
                        const string ValueName = "AutoEndTasks";

                        // レジストリ・キーのパスを指定してレジストリを開く
                        using (var key = Registry.CurrentUser.OpenSubKey(KeyName))
                        {
                            // レジストリの値を取得
                            if (key != null)
                            {
                                var tmpValue = key.GetValue(ValueName);
                                if (tmpValue != null)
                                {
                                    regValue = (string)tmpValue;
                                }
                            }
                        }
                    }
                    catch
                    {
                    }

                    if (int.TryParse(regValue, out reloadWaitTimeMs) == false)
                    {
                        reloadWaitTimeMs = 5000;
                    }
                }

                return reloadWaitTimeMs;
            }
        }

        /// <summary>
        /// 監視開始
        /// </summary>
        public void Start()
        {
            this.watcher.EnableRaisingEvents = true;
        }

        /// <summary>
        /// 監視終了
        /// </summary>
        public void Stop()
        {
            this.watcher.EnableRaisingEvents = false;
        }

        /// <summary>
        /// アンマネージ リソースの解放およびリセットに関連付けられているアプリケーション定義のタスクを実行します。
        /// </summary>
        public void Dispose()
        {
            if (this.watcher == null)
            {
                return;
            }

            this.throttleExecuter.Dispose();

            this.Stop();

            this.watcher.Changed -= this.Changed;
            this.watcher.Created -= this.Changed;
            this.watcher.Deleted -= this.Changed;
            this.watcher.Renamed -= this.Changed;

            this.watcher.EnableRaisingEvents = false;
            this.watcher.Dispose();
            this.watcher = null;
        }

        /// <summary>
        /// ファイル変更時の処理
        /// </summary>
        /// <param name="sender">呼び元</param>
        /// <param name="args">呼び元からの情報</param>
        private void Changed(object sender, FileSystemEventArgs args)
        {
            if (this.watcher == null)
            {
                return;
            }

            var retryCount = ReloadReloadWaitTimeMs / 100;
            var isDirectory = Directory.Exists(args.FullPath);

            // 稀にファイルハンドルの競合が起こるので、他のアプリが手放すのを待つ
            for (var i = 0; i != retryCount; ++i)
            {
                // ファイルの空読みを行って例外が起きなければ競合なしとする
                try
                {
                    if (isDirectory)
                    {
                        Directory.EnumerateFiles(args.FullPath).Any();
                        break;
                    }
                    else
                    {
                        using (var stream = File.Open(args.FullPath, FileMode.Open, FileAccess.Read))
                        {
                            break;
                        }
                    }
                }
                catch (Exception e)
                {
                    if (e is FileNotFoundException)
                    {
                        break;
                    }
                    else if (e is DirectoryNotFoundException)
                    {
                        break;
                    }

                    Thread.Sleep(100);
                    Logger.Log(LogLevels.Debug, "FileMonitor.Changed : watcherChanged " + e.Message);
                }
            }

            this.throttleExecuter.Execute(
                100,
                () =>
                {
                    // イベント発生
                    if (this.FileChanged != null)
                    {
                        this.FileChanged(this, args);
                    }
                });
        }
    }
}
