﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.Foundation.Command
{
    /// <summary>
    /// コマンドのための抽象基底クラス.
    /// </summary>
    public abstract class CommandBase
    {
        /// <summary>
        /// コンストラクタ.
        /// </summary>
        public CommandBase()
        {
            this.IsInitialized = false;
            this.KeepExecuting = false;
            this.CanUndo = true;
        }

        /// <summary>
        /// コマンドが初期化済みか示すフラグ.
        /// </summary>
        public virtual bool IsInitialized { get; set; }

        /// <summary>
        /// コマンドの実行を継続するか示すフラグ.
        /// </summary>
        public virtual bool KeepExecuting { get; set; }

        /// <summary>
        /// コマンドの取り消しが可能かどうか示すフラグ.
        /// </summary>
        public virtual bool CanUndo { get; set; }

        /// <summary>
        /// 初期化.
        /// </summary>
        /// <param name="args">データモデルと変更する値を含むオブジェクトの配列.</param>
        /// <returns>True on success.</returns>
        public abstract bool Initialize(object[] args);

        /// <summary>
        /// ターゲットを取得する.
        /// </summary>
        /// <returns>ターゲットのリファレンス.</returns>
        public abstract object GetTarget();

        /// <summary>
        /// コマンドを実行前に呼び出すメソッド.
        /// </summary>
        /// <param name="type">実行タイプ.</param>
        /// <returns>True on success.</returns>
        public virtual bool PrepareExecution(ExecutionType type)
        {
            return true;
        }

        /// <summary>
        /// 実行するコマンド.
        /// </summary>
        /// <returns>True on success.</returns>
        public abstract bool Do();

        /// <summary>
        /// コマンド取り消し.
        /// </summary>
        /// <returns>True on success.</returns>
        public abstract bool Undo();

        /// <summary>
        /// コマンド再実行.
        /// </summary>
        /// <returns>True on success.</returns>
        public virtual bool Redo()
        {
            return this.Do();
        }

        /// <summary>
        /// コマンド実行後に呼び出すメソッド.
        /// </summary>
        /// <param name="type">実行タイプ.</param>
        public virtual void FinalizeExecution(ExecutionType type)
        {
        }

        /// <summary>
        /// Find the command stack target from any IHierarchyObject instance.
        /// </summary>
        /// <param name="obj">The IHierarchyObject instance from where
        /// to look for the command stack target.</param>
        /// <returns>Returns the proper command stack target if found,
        /// or the IHierarchyObject object given as parameter otherwise.</returns>
        public static IHierarchyObject FindCommandStackTargetFromHierarchyObject(IHierarchyObject obj)
        {
            IHierarchyObject cmdStackTarget = obj;

            while (cmdStackTarget != null)
            {
                var attributes = cmdStackTarget.GetType()
                    .GetCustomAttributes(typeof(CommandStackOwnerAttribute), false);
                bool result = attributes.Length > 0 &&
                    attributes[0] is CommandStackOwnerAttribute;

                if (result)
                {
                    break;
                }

                cmdStackTarget = cmdStackTarget.Parent;
            }

            return cmdStackTarget;
        }
    }
}
