﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Core;
using EffectMaker.Foundation.Extensions;

namespace EffectMaker.Foundation.Collections.Generic
{
    /// <summary>
    /// An observable collection that can sort by type.
    /// </summary>
    /// <typeparam name="T">Type of item.</typeparam>
    public class TypeOrderedObservableCollection<T> : ObservableCollection<T>
    {
        /// <summary>
        /// Flag that tells whether the ordering should be performed when adding an item or not.
        /// </summary>
        private bool allowOrdering = true;

        /// <summary>
        /// Gets or sets the types that define the type order.
        /// </summary>
        public Type[] TypeOrder { get; set; }

        /// <summary>
        /// Suspends the type ordering when an item is added.
        /// </summary>
        public void SuspendTypeOrdering()
        {
            this.allowOrdering = false;
        }

        /// <summary>
        /// Resumes the type ordering when an item is added.
        /// </summary>
        public void ResumeTypeOrdering()
        {
            this.allowOrdering = true;
        }

        /// <summary>
        /// Sorts the list according to the type order.
        /// Warning: This clears the list and re-add all items sorted,
        /// so it might generate undesirable INotifyCollectionChanged event.
        /// </summary>
        public void Sort()
        {
            var list = new List<T>(this.Items);

            list.Sort(new TypeOrderingComparer<T>(this.TypeOrder));

            this.SuspendTypeOrdering();

            try
            {
                this.Clear();
                foreach (T item in list)
                {
                    this.Add(item);
                }
            }
            finally
            {
                this.ResumeTypeOrdering();
            }
        }

        /// <summary>
        /// 複数のアイテムを一気に追加します。
        /// </summary>
        /// <param name="list">一気に追加したいアイテムが入ったコレクション</param>
        public void AddRange(IEnumerable<T> list)
        {
            foreach (T item in list)
            {
                this.Items.Add(item);
            }

            this.OnCollectionChanged(new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, (IList)list));
        }

        /// <summary>
        /// Called when an item is added or inserted.
        /// </summary>
        /// <param name="index">The original index where the item is added.</param>
        /// <param name="item">The added item.</param>
        protected override void InsertItem(int index, T item)
        {
            if (this.allowOrdering && this.TypeOrder != null && item != null)
            {
                index = this.FindLastIndexOf(item.GetType());
            }

            base.InsertItem(index, item);
        }

        /// <summary>
        /// Moves an item. Not supported.
        /// </summary>
        /// <param name="oldIndex">Old index.</param>
        /// <param name="newIndex">New index.</param>
        protected override void MoveItem(int oldIndex, int newIndex)
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// Sets an item at a given index. Not supported.
        /// </summary>
        /// <param name="index">Index where to add the item.</param>
        /// <param name="item">Item to add.</param>
        protected override void SetItem(int index, T item)
        {
            throw new NotSupportedException();
        }

        /// <summary>
        /// Finds the index of the most matching type in the type ordering array.
        /// </summary>
        /// <param name="specificInstance">Instance to find the most matching type.</param>
        /// <returns>Returns the index of the most matching type, -1 if not found.</returns>
        private int GetGenericTypeIndex(T specificInstance)
        {
            if (specificInstance == null)
            {
                return -1;
            }

            return this.GetGenericTypeIndex(specificInstance.GetType());
        }

        /// <summary>
        /// Finds the index of the most matching type in the type ordering array.
        /// </summary>
        /// <param name="specificType">Type to find the most matching type.</param>
        /// <returns>Returns the index of the most matching type, -1 if not found.</returns>
        private int GetGenericTypeIndex(Type specificType)
        {
            return this.TypeOrder.FindIndex(t => t.IsAssignableFrom(specificType));
        }

        /// <summary>
        /// Finds the last index of the most matching instance in the current list.
        /// </summary>
        /// <param name="type">Type to find the last item.</param>
        /// <returns>Returns the index of the last item that most match the given type,
        /// -1 if not found.</returns>
        private int FindLastIndexOf(Type type)
        {
            int typeIndex = this.TypeOrder.FindIndex(t => t.IsAssignableFrom(type));

            if (typeIndex == -1)
            {
                return this.Items.Count;
            }
            else
            {
                int[] indices = this.Items
                    .Select(this.GetGenericTypeIndex)
                    .ToArray();

                int index = 0;

                while (index < indices.Length && typeIndex >= indices[index])
                {
                    index++;
                }

                return index;
            }
        }
    }
}
