﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace EffectMaker.Foundation.Collections.Generic
{
    /// <summary>
    /// Generic tree data structure class.
    /// </summary>
    /// <typeparam name="TValue">The value type of the tree nodes.</typeparam>
    public class Tree<TValue> : IEnumerable<TValue>
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public Tree()
        {
        }

        /// <summary>
        /// Get or set the root tree node.
        /// </summary>
        public TreeNode<TValue> Root { get; set; }

        /// <summary>
        /// Get an enumerator to traverse the whole tree.
        /// </summary>
        /// <returns>The enumerator.</returns>
        public IEnumerator<TValue> GetEnumerator()
        {
            return new TreeEnumerator(this.Root);
        }

        /// <summary>
        /// Get an enumerator to traverse the whole tree.
        /// </summary>
        /// <returns>The primitive enumerator.</returns>
        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return new TreeEnumerator(this.Root);
        }

        /// <summary>
        /// Get an enumerator to traverse a sub tree.
        /// </summary>
        /// <param name="root">The root node of the sub tree to start traversing.</param>
        /// <returns>The enumerator.</returns>
        public IEnumerator<TValue> GetEnumerator(TreeNode<TValue> root)
        {
            return new TreeEnumerator(root);
        }

        /// <summary>
        /// Enumerator class for the tree.
        /// </summary>
        public class TreeEnumerator : IEnumerator<TValue>
        {
            /// <summary>The root node to start the enumeration.</summary>
            private TreeNode<TValue> enumerationRoot = null;

            /// <summary>The current enumerating tree node.</summary>
            private TreeNode<TValue> currEnumeratingNode = null;

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="root">The root node to start enumerating.</param>
            public TreeEnumerator(TreeNode<TValue> root)
            {
                this.enumerationRoot = root;
            }

            /// <summary>
            /// Get the current item.
            /// </summary>
            object System.Collections.IEnumerator.Current
            {
                get { return this.Current; }
            }

            /// <summary>
            /// Get the current item with type.
            /// </summary>
            public TValue Current
            {
                get { return this.currEnumeratingNode.Data; }
            }

            /// <summary>
            /// Dispose the enumerator.
            /// </summary>
            public void Dispose()
            {
                this.Reset();
            }

            /// <summary>
            /// Move to the next tree node.
            /// The first leaf child of the root will be returned first,
            /// then the leaf child of the next sibling if any, then the
            /// parent of the child.
            /// The root it self will be the last one visited.
            /// </summary>
            /// <returns>True on success.</returns>
            public bool MoveNext()
            {
                if (this.enumerationRoot == this.currEnumeratingNode)
                {
                    return false;
                }

                if (this.currEnumeratingNode == null)
                {
                    this.currEnumeratingNode = this.enumerationRoot;

                    // Internally, we start from the first leaf child of the root.
                    while (this.currEnumeratingNode.FirstChild != null)
                    {
                        this.currEnumeratingNode = this.currEnumeratingNode.FirstChild;
                    }
                }
                else
                {
                    TreeNode<TValue> nextNode = this.TraverseChildren(this.currEnumeratingNode);
                    if (nextNode == null)
                    {
                        return false;
                    }
                    else
                    {
                        this.currEnumeratingNode = nextNode;
                    }
                }

                return true;
            }

            /// <summary>
            /// Reset the enumerator.
            /// </summary>
            public void Reset()
            {
                this.currEnumeratingNode = this.enumerationRoot;
            }

            /// <summary>
            /// Traverse the children of the given tree node.
            /// </summary>
            /// <param name="lastNode">The last node traversed.</param>
            /// <returns>The current tree node.</returns>
            private TreeNode<TValue> TraverseChildren(TreeNode<TValue> lastNode)
            {
                TreeNode<TValue> currNode = null;

                if (lastNode.NextSibling != null)
                {
                    // Find the next sibling.
                    currNode = lastNode.NextSibling;

                    // Get to the first leaf child of the sibling.
                    if (currNode.FirstChild != null)
                    {
                        currNode = currNode.FirstChild;
                        while (currNode.FirstChild != null)
                        {
                            currNode = currNode.FirstChild;
                        }
                    }
                }
                else if (lastNode.Parent != null)
                {
                    // Find the parent.
                    currNode = lastNode.Parent;
                }

                return currNode;
            }
        }
    }

    /// <summary>
    /// Generic tree node class.
    /// </summary>
    /// <typeparam name="TValue">The value type of the tree node.</typeparam>
    public class TreeNode<TValue>
    {
        /// <summary>The tree for this tree node.</summary>
        private Tree<TValue> tree = null;

        /// <summary>The parent tree node.</summary>
        private TreeNode<TValue> parent = null;

        /// <summary>The next sibling.</summary>
        private TreeNode<TValue> nextSibling = null;

        /// <summary>The previous sibling.</summary>
        private TreeNode<TValue> prevSibling = null;

        /// <summary>The first child tree node.</summary>
        private TreeNode<TValue> firstChild = null;

        /// <summary>The data of the tree node.</summary>
        private TValue data;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="tree">The tree of the tree node.</param>
        public TreeNode(Tree<TValue> tree)
        {
            this.tree = tree;
        }

        /// <summary>
        /// Get the parent of the node.
        /// </summary>
        public TreeNode<TValue> Parent
        {
            get { return this.parent; }
        }

        /// <summary>
        /// Get the next sibling.
        /// </summary>
        public TreeNode<TValue> NextSibling
        {
            get { return this.nextSibling; }
        }

        /// <summary>
        /// Get the previous sibling.
        /// </summary>
        public TreeNode<TValue> PrevSibling
        {
            get { return this.prevSibling; }
        }

        /// <summary>
        /// Get the first child.
        /// </summary>
        public TreeNode<TValue> FirstChild
        {
            get { return this.firstChild; }
        }

        /// <summary>
        /// Get the last child.
        /// </summary>
        /// <remarks>
        /// This property gets the first child then loop through all it's siblings,
        /// hence calling this property too often could introduce a performance overhead.
        /// </remarks>
        public TreeNode<TValue> LastChild
        {
            get
            {
                // Get the first child
                TreeNode<TValue> child = this.FirstChild;
                if (child == null)
                {
                    return null;
                }

                // Loop through the children to get their last sibling.
                while (child.NextSibling != null)
                {
                    child = child.NextSibling;
                }

                return child;
            }
        }

        /// <summary>
        /// Get or set the data of the tree node.
        /// </summary>
        public TValue Data
        {
            get { return this.data; }
            set { this.data = value; }
        }

        /// <summary>
        /// Add a child to the back of the last child.
        /// </summary>
        /// <param name="data">The data for the added child.</param>
        /// <returns>The added child node, null is returned on fail.</returns>
        public TreeNode<TValue> AddChild(TValue data)
        {
            TreeNode<TValue> child = null;

            // Has the node has any child yet?
            if (this.FirstChild == null)
            {
                // Create the child.
                child = new TreeNode<TValue>(this.tree) { data = data };

                // Setup the links.
                this.firstChild = child;
                child.parent = this;
            }
            else
            {
                // Create the child.
                child = new TreeNode<TValue>(this.tree) { data = data };

                // Find the last child.
                TreeNode<TValue> lastChild = this.LastChild;

                // Setup the links.
                lastChild.nextSibling = child;
                child.prevSibling = lastChild;
                child.parent = this;
            }

            return child;
        }

        /// <summary>
        /// Add a child to the back of the last child.
        /// </summary>
        /// <param name="childNode">The child tree node to add.</param>
        /// <returns>The added child node, null is returned on fail.</returns>
        public TreeNode<TValue> AddChild(TreeNode<TValue> childNode)
        {
            if (childNode == null)
            {
                return null;
            }

            // Has the node has any child yet?
            if (this.FirstChild == null)
            {
                // Setup the links.
                this.firstChild = childNode;
                childNode.parent = this;
                childNode.tree = this.tree;
            }
            else
            {
                // Find the last child.
                TreeNode<TValue> lastChild = this.LastChild;

                // Setup the links.
                lastChild.nextSibling = childNode;
                childNode.prevSibling = lastChild;
                childNode.parent = this;
                childNode.tree = this.tree;
            }

            return childNode;
        }

        /// <summary>
        /// Remove this tree node from the tree.
        /// </summary>
        /// <returns>The removed tree node.</returns>
        public TreeNode<TValue> Remove()
        {
            // Is this the first child of the parent?
            if (this.prevSibling == null && this.parent != null)
            {
                this.parent.firstChild = this.nextSibling;
            }

            // Connect the next sibling to the previous.
            if (this.prevSibling != null)
            {
                this.prevSibling.nextSibling = this.nextSibling;
            }

            // Connect the previous sibling to the next.
            if (this.nextSibling != null)
            {
                this.nextSibling.prevSibling = this.prevSibling;
            }

            // If this node is the root, clean up the reference from the tree.
            if (this.tree != null && this.tree.Root == this)
            {
                this.tree.Root = null;
            }

            // Clean up the connections.
            this.parent = null;
            this.prevSibling = null;
            this.nextSibling = null;
            this.tree = null;

            return this;
        }
    }
}
