﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

namespace EffectMaker.Foundation.Collections.Generic
{
    /// <summary>
    /// LRUキャッシュです。
    /// LruCacheValue クラスを継承して GetValueSize メソッドや DisposeValue メソッドを
    /// オーバーライドすることで、キャッシュデータに応じた機能拡張ができます。
    /// </summary>
    /// <typeparam name="TKey">キーの型</typeparam>
    /// <typeparam name="TValue">値の型</typeparam>
    public class LruCacheSpecial<TKey, TValue> : LruCache<TKey, TValue>
        where TValue : LruCacheSpecialValue
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="maxCapacity">最大キャパシティ</param>
        /// <param name="comparer">キーの比較時に使用する IEqualityComparer 実装</param>
        public LruCacheSpecial(int maxCapacity, IEqualityComparer<TKey> comparer = null)
            : base(maxCapacity, comparer)
        {
        }

        /// <summary>
        /// 値のサイズを取得します。
        /// </summary>
        /// <param name="value">値</param>
        /// <returns>値のサイズを返します。</returns>
        protected override int GetValueSize(TValue value)
        {
            return value.GetValueSize();
        }

        /// <summary>
        /// 値の解放時の処理を行います。
        /// </summary>
        /// <param name="key">キー</param>
        /// <param name="value">値</param>
        protected override void OnDisposingValue(TKey key, TValue value)
        {
            value.DisposingValue();
        }
    }

    /// <summary>
    /// キャッシュ値です。
    /// </summary>
    public class LruCacheSpecialValue
    {
        /// <summary>
        /// 値のサイズを取得します。
        /// </summary>
        public virtual int GetValueSize()
        {
            return 1;
        }

        /// <summary>
        /// 値の解放時の処理を行います。
        /// </summary>
        public virtual void DisposingValue()
        {
        }
    }
}
