﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.Foundation.Animation
{
    #region Event arguments for animation update event

    /// <summary>
    /// Event argument class for update animation events.
    /// </summary>
    public class AnimationUpdatedEventArgs : EventArgs
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="animationValue">The updated animation value.</param>
        public AnimationUpdatedEventArgs(object animationValue)
            : base()
        {
            this.Value = animationValue;
        }

        /// <summary>
        /// The updated animation value.
        /// </summary>
        public object Value { get; private set; }
    }

    #endregion

    #region Base animation controller (abstract class)

    /// <summary>
    /// Class for controlling animation.
    /// </summary>
    /// <typeparam name="TValue">The animation value type.</typeparam>
    public abstract class AnimationController<TValue>
    {
        /// <summary>Timer for animation.</summary>
        private System.Threading.Timer timer = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public AnimationController()
        {
            this.AnimationStartTime = DateTime.MinValue;
            this.AnimationLength = 0.0;
            this.AnimationTime = 0.0;
        }

        /// <summary>
        /// Event triggered when the animation value is updated.
        /// </summary>
        public event EventHandler<AnimationUpdatedEventArgs> AnimationUpdated = null;

        /// <summary>
        /// Get the flag indicating whether the animation is playing.
        /// </summary>
        public bool IsAnimationPlaying { get; private set; }

        /// <summary>
        /// Get the initial value of the animation.
        /// </summary>
        public TValue AnimationInitialValue { get; private set; }

        /// <summary>
        /// Get the target value of the animation.
        /// </summary>
        public TValue AnimationTargetValue { get; private set; }

        /// <summary>
        /// Get the current animation value.
        /// </summary>
        public TValue AnimationValue { get; private set; }

        /// <summary>
        /// Get the start time of the animation.
        /// </summary>
        public DateTime AnimationStartTime { get; private set; }

        /// <summary>
        /// Get the length of the animation in seconds.
        /// </summary>
        public double AnimationLength { get; private set; }

        /// <summary>
        /// Get the elapsed time of the animation in seconds.
        /// </summary>
        public double AnimationTime { get; private set; }

        /// <summary>
        /// Start a new animation.
        /// </summary>
        /// <param name="animationLength">The length of the animation in seconds.</param>
        /// <param name="framesPerSecond">The number of animation frames per second.</param>
        /// <param name="initialValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        public void StartAnimation(
            double animationLength,
            double framesPerSecond,
            TValue initialValue,
            TValue targetValue,
            TValue currValue)
        {
            // First dispose the timer if it exists.
            if (this.timer != null)
            {
                this.timer.Dispose();
                this.timer = null;
            }

            // Setup animation parameters.
            this.AnimationLength = animationLength;

            double timerInterval = 1000.0 / framesPerSecond;
            double timerDueTime = 0.0;

            if (initialValue is ICloneable)
            {
                this.AnimationInitialValue = (TValue)((ICloneable)initialValue).Clone();
                this.AnimationTargetValue = (TValue)((ICloneable)targetValue).Clone();
            }
            else
            {
                this.AnimationInitialValue = initialValue;
                this.AnimationTargetValue = targetValue;
            }

            // Compute the current animation progress with the given animation values.
            double progress = this.ComputeProgress(initialValue, targetValue, currValue);
            if (progress <= 0.0)
            {
                this.AnimationStartTime = DateTime.Now;
                this.AnimationTime = 0.0;
            }
            else if (progress >= 1.0)
            {
                this.AnimationStartTime = DateTime.Now - TimeSpan.FromSeconds(this.AnimationLength);
                this.AnimationTime = this.AnimationLength;
            }
            else
            {
                double elapsedTime = this.AnimationLength * progress;

                this.AnimationStartTime = DateTime.Now - TimeSpan.FromSeconds(elapsedTime);
                this.AnimationTime = elapsedTime;
            }

            // Update animation value.
            TValue animValue = this.UpdateAnimationValue(
                                                         progress,
                                                         this.AnimationInitialValue,
                                                         this.AnimationTargetValue);

            // Set the animation value.
            if (animValue is ISettable)
            {
                ((ISettable)this.AnimationValue).Set(animValue);
            }
            else
            {
                this.AnimationValue = animValue;
            }

            if (progress >= 1.0)
            {
                // The animation has already finished, just stop the animation.
                this.StopAnimation();
                return;
            }

            // Setup animation playback flag.
            this.IsAnimationPlaying = true;

            // Setup the timer.
            this.timer = new System.Threading.Timer(
                                                    this.HandleTimerCallback,
                                                    null,
                                                    (int)timerDueTime,
                                                    (int)timerInterval);
        }

        /// <summary>
        /// Start a new animation.
        /// </summary>
        /// <param name="animationLength">The length of the animation in seconds.</param>
        /// <param name="framesPerSecond">The number of animation frames per second.</param>
        /// <param name="initialValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        public void StartAnimation(
            double animationLength,
            double framesPerSecond,
            TValue initialValue,
            TValue targetValue)
        {
            // First dispose the timer if it exists.
            if (this.timer != null)
            {
                this.timer.Dispose();
                this.timer = null;
            }

            // Setup animation parameters.
            this.AnimationStartTime = DateTime.Now;
            this.AnimationLength = animationLength;
            this.AnimationTime = 0.0;

            if (initialValue is ICloneable)
            {
                this.AnimationInitialValue = (TValue)((ICloneable)initialValue).Clone();
                this.AnimationTargetValue = (TValue)((ICloneable)targetValue).Clone();
                this.AnimationValue = (TValue)((ICloneable)initialValue).Clone();
            }
            else
            {
                this.AnimationInitialValue = initialValue;
                this.AnimationTargetValue = targetValue;
                this.AnimationValue = initialValue;
            }

            // Check animation length.
            if (this.AnimationLength <= 0.0)
            {
                // There is no animation needed, just stop the animation.
                this.StopAnimation();
                return;
            }

            // Setup animation playback flag.
            this.IsAnimationPlaying = true;

            // Setup the timer.
            this.timer = new System.Threading.Timer(
                                                    this.HandleTimerCallback,
                                                    null,
                                                    0,
                                                    (int)(1000.0 / framesPerSecond));
        }

        /// <summary>
        /// Stop the animation.
        /// The animation can be started again from the current states with ResumeAnimation().
        /// </summary>
        public void StopAnimation()
        {
            this.IsAnimationPlaying = false;

            if (this.timer != null)
            {
                this.timer.Dispose();
                this.timer = null;
            }

            // Fire the event.
            if (this.AnimationUpdated != null)
            {
                this.AnimationUpdated(this, new AnimationUpdatedEventArgs(this.AnimationValue));
            }
        }

        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected abstract TValue UpdateAnimationValue(
            double t,
            TValue initValue,
            TValue targetValue);

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected abstract double ComputeProgress(
            TValue initValue,
            TValue targetValue,
            TValue currValue);

        /// <summary>
        /// Handle timer callback.
        /// </summary>
        /// <param name="state">
        /// An object containing application-specific information
        /// relevant to the method invoked by this delegate, or null.
        /// </param>
        private void HandleTimerCallback(object state)
        {
            // Compute the current animation time in seconds.
            this.AnimationTime = (DateTime.Now - this.AnimationStartTime).TotalSeconds;

            // Compute the animation progress.
            double t = this.AnimationTime / this.AnimationLength;
            if (t > 1.0)
            {
                this.StopAnimation();
                return;
            }

            // Update animation value.
            TValue updatedValue = this.UpdateAnimationValue(
                                                            Math.Min(1.0f, t),
                                                            this.AnimationInitialValue,
                                                            this.AnimationTargetValue);

            // Set the updated value.
            if (updatedValue is ISettable)
            {
                ((ISettable)this.AnimationValue).Set(updatedValue);
            }
            else
            {
                this.AnimationValue = updatedValue;
            }

            // Fire the event.
            if (this.AnimationUpdated != null)
            {
                this.AnimationUpdated(this, new AnimationUpdatedEventArgs(this.AnimationValue));
            }
        }
    }

    #endregion

    #region Animation controller (double)

    /// <summary>
    /// Animation controller for double precision floating point animation value.
    /// </summary>
    public class AnimationControllerD : AnimationController<double>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override double UpdateAnimationValue(
            double t,
            double initValue,
            double targetValue)
        {
            return ((targetValue - initValue) * t) + initValue;
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            double initValue,
            double targetValue,
            double currValue)
        {
            return (currValue - initValue) / (targetValue - initValue);
        }
    }

    #endregion

    #region Animation controller (float)

    /// <summary>
    /// Animation controller for floating point animation value.
    /// </summary>
    public class AnimationControllerF : AnimationController<float>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override float UpdateAnimationValue(
            double t,
            float initValue,
            float targetValue)
        {
            return ((targetValue - initValue) * (float)t) + initValue;
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            float initValue,
            float targetValue,
            float currValue)
        {
            return (double)(currValue - initValue) / (double)(targetValue - initValue);
        }
    }

    #endregion

    #region Animation controller (int)

    /// <summary>
    /// Animation controller for integer animation value.
    /// </summary>
    public class AnimationControllerI : AnimationController<int>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override int UpdateAnimationValue(
            double t,
            int initValue,
            int targetValue)
        {
            return (int)Math.Round((double)(targetValue - initValue) * t) + initValue;
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            int initValue,
            int targetValue,
            int currValue)
        {
            return (double)(currValue - initValue) / (double)(targetValue - initValue);
        }
    }

    #endregion

    #region Animation controller (uint)

    /// <summary>
    /// Animation controller for unsigned integer animation value.
    /// </summary>
    public class AnimationControllerU : AnimationController<uint>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override uint UpdateAnimationValue(
            double t,
            uint initValue,
            uint targetValue)
        {
            double value = (((double)targetValue - (double)initValue) * t) + (double)initValue;
            if (value < (double)uint.MinValue)
            {
                return uint.MinValue;
            }
            else if (value > (double)uint.MaxValue)
            {
                return uint.MaxValue;
            }
            else
            {
                return (uint)value;
            }
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            uint initValue,
            uint targetValue,
            uint currValue)
        {
            return ((double)currValue - (double)initValue) / ((double)targetValue - (double)initValue);
        }
    }

    #endregion

    #region Animation controller (byte)

    /// <summary>
    /// Animation controller for byte animation value.
    /// </summary>
    public class AnimationControllerB : AnimationController<byte>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override byte UpdateAnimationValue(
            double t,
            byte initValue,
            byte targetValue)
        {
            double value = (((double)targetValue - (double)initValue) * t) + (double)initValue;
            if (value < (double)byte.MinValue)
            {
                return byte.MinValue;
            }
            else if (value > (double)byte.MaxValue)
            {
                return byte.MaxValue;
            }
            else
            {
                return (byte)value;
            }
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            byte initValue,
            byte targetValue,
            byte currValue)
        {
            return ((double)currValue - (double)initValue) / ((double)targetValue - (double)initValue);
        }
    }

    #endregion

    #region Animation controller (Primitived)

    /// <summary>
    /// Animation controller for double precision floating point primitive animation value.
    /// (Vector1d, Vector2d, Vector3d, Vector4d)
    /// </summary>
    public class AnimationControllerPrimitiveD : AnimationController<Primitived>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override Primitived UpdateAnimationValue(
            double t,
            Primitived initValue,
            Primitived targetValue)
        {
            var currValue = initValue.Clone() as Primitived;
            for (int i = 0; i < initValue.Count; ++i)
            {
                currValue[i] = ((targetValue[i] - initValue[i]) * t) + initValue[i];
            }

            return currValue;
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            Primitived initValue,
            Primitived targetValue,
            Primitived currValue)
        {
            double progress = 1.0;
            for (int i = 0; i < initValue.Count; ++i)
            {
                progress = Math.Min(
                    progress,
                    (currValue[i] - initValue[i]) / (targetValue[i] - initValue[i]));
            }

            return progress;
        }
    }

    #endregion

    #region Animation controller (Primitivef)

    /// <summary>
    /// Animation controller for floating point primitive animation value.
    /// (Vector1f, Vector2f, Vector3f, Vector4f, ColorRgba)
    /// </summary>
    public class AnimationControllerPrimitiveF : AnimationController<Primitivef>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override Primitivef UpdateAnimationValue(
            double t,
            Primitivef initValue,
            Primitivef targetValue)
        {
            var currValue = initValue.Clone() as Primitivef;
            for (int i = 0; i < currValue.Count; ++i)
            {
                currValue[i] = ((targetValue[i] - initValue[i]) * (float)t) + initValue[i];
            }

            return currValue;
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            Primitivef initValue,
            Primitivef targetValue,
            Primitivef currValue)
        {
            double progress = 1.0;
            for (int i = 0; i < initValue.Count; ++i)
            {
                progress = Math.Min(
                    progress,
                    (double)(currValue[i] - initValue[i]) / (double)(targetValue[i] - initValue[i]));
            }

            return progress;
        }
    }

    #endregion

    #region Animation controller (Primitivei)

    /// <summary>
    /// Animation controller for integer primitive animation value.
    /// (Vector1i, Vector2i, Vector3i, Vector4i)
    /// </summary>
    public class AnimationControllerPrimitiveI : AnimationController<Primitivei>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override Primitivei UpdateAnimationValue(
            double t,
            Primitivei initValue,
            Primitivei targetValue)
        {
            var currValue = initValue.Clone() as Primitivei;
            for (int i = 0; i < currValue.Count; ++i)
            {
                currValue[i] =
                    (int)Math.Round((double)(targetValue[i] - initValue[i]) * t) + initValue[i];
            }

            return currValue;
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            Primitivei initValue,
            Primitivei targetValue,
            Primitivei currValue)
        {
            double progress = 1.0;
            for (int i = 0; i < initValue.Count; ++i)
            {
                progress = Math.Min(
                    progress,
                    ((double)currValue[i] - (double)initValue[i]) / ((double)targetValue[i] - (double)initValue[i]));
            }

            return progress;
        }
    }

    #endregion

    #region Animation controller (Primiitveu)

    /// <summary>
    /// Animation controller for unsigned integer primitive animation value.
    /// (Vector1u, Vector2u, Vector3u, Vector4u)
    /// </summary>
    public class AnimationControllerPrimitiveU : AnimationController<Primitiveu>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override Primitiveu UpdateAnimationValue(
            double t,
            Primitiveu initValue,
            Primitiveu targetValue)
        {
            var currValue = initValue.Clone() as Primitiveu;
            for (int i = 0; i < currValue.Count; ++i)
            {
                double value =
                    (((double)targetValue[i] - (double)initValue[i]) * t) + (double)initValue[i];
                if (value < (double)uint.MinValue)
                {
                    currValue[i] = uint.MinValue;
                }
                else if (value > (double)uint.MaxValue)
                {
                    currValue[i] = uint.MaxValue;
                }
                else
                {
                    currValue[i] = (uint)value;
                }
            }

            return currValue;
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            Primitiveu initValue,
            Primitiveu targetValue,
            Primitiveu currValue)
        {
            double progress = 1.0;
            for (int i = 0; i < initValue.Count; ++i)
            {
                progress = Math.Min(
                    progress,
                    ((double)currValue[i] - (double)initValue[i]) / ((double)targetValue[i] - (double)initValue[i]));
            }

            return progress;
        }
    }

    #endregion

    #region Animation controller (System.Drawing.Color)

    /// <summary>
    /// Animation controller for windows color.
    /// </summary>
    public class AnimationControllerColor : AnimationController<System.Drawing.Color>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override Color UpdateAnimationValue(
            double t,
            Color initValue,
            Color targetValue)
        {
            return Color.FromArgb(
                                  this.InterpolateColorElement(t, initValue.A, targetValue.A),
                                  this.InterpolateColorElement(t, initValue.R, targetValue.R),
                                  this.InterpolateColorElement(t, initValue.G, targetValue.G),
                                  this.InterpolateColorElement(t, initValue.B, targetValue.B));
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            Color initValue,
            Color targetValue,
            Color currValue)
        {
            double progress = 1.0;
            progress = Math.Min(progress, ((double)currValue.A - (double)initValue.A) / ((double)targetValue.A - (double)initValue.A));
            progress = Math.Min(progress, ((double)currValue.R - (double)initValue.R) / ((double)targetValue.R - (double)initValue.R));
            progress = Math.Min(progress, ((double)currValue.G - (double)initValue.G) / ((double)targetValue.G - (double)initValue.G));
            progress = Math.Min(progress, ((double)currValue.B - (double)initValue.B) / ((double)targetValue.B - (double)initValue.B));

            return progress;
        }

        /// <summary>
        /// Interpolate between two color element values.
        /// </summary>
        /// <param name="t">Interpolation factor.</param>
        /// <param name="value1">Value 1.</param>
        /// <param name="value2">Value 2.</param>
        /// <returns>The interpolated value.</returns>
        private byte InterpolateColorElement(
            double t,
            byte value1,
            byte value2)
        {
            double value =
                (((double)value2 - (double)value1) * t) + (double)value1;
            if (value < (double)byte.MinValue)
            {
                return byte.MinValue;
            }
            else if (value > (double)byte.MaxValue)
            {
                return byte.MaxValue;
            }
            else
            {
                return (byte)value;
            }
        }
    }

    #endregion

    #region Animation controller (System.Drawing.Point)

    /// <summary>
    /// Animation controller for windows point.
    /// </summary>
    public class AnimationControllerPoint : AnimationController<Point>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override Point UpdateAnimationValue(
            double t,
            Point initValue,
            Point targetValue)
        {
            return new Point(
                (int)Math.Round((double)(targetValue.X - initValue.X) * t) + initValue.X,
                (int)Math.Round((double)(targetValue.Y - initValue.Y) * t) + initValue.Y);
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            Point initValue,
            Point targetValue,
            Point currValue)
        {
            double progress = 1.0;
            progress = Math.Min(progress, ((double)currValue.X - (double)initValue.X) / ((double)targetValue.X - (double)initValue.X));
            progress = Math.Min(progress, ((double)currValue.Y - (double)initValue.Y) / ((double)targetValue.Y - (double)initValue.Y));

            return progress;
        }
    }

    #endregion

    #region Animation controller (System.Drawing.Point)

    /// <summary>
    /// Animation controller for windows point.
    /// </summary>
    public class AnimationControllerPointF : AnimationController<PointF>
    {
        /// <summary>
        /// Update animation value.
        /// </summary>
        /// <param name="t">The progress of the animation from 0.0 to 1.0</param>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <returns>The updated animation value.</returns>
        protected override PointF UpdateAnimationValue(
            double t,
            PointF initValue,
            PointF targetValue)
        {
            return new PointF(
                ((targetValue.X - initValue.X) * (float)t) + initValue.X,
                ((targetValue.Y - initValue.Y) * (float)t) + initValue.Y);
        }

        /// <summary>
        /// Compute animation progress with the animation values.
        /// </summary>
        /// <param name="initValue">The initial animation value.</param>
        /// <param name="targetValue">The target animation value.</param>
        /// <param name="currValue">The current animation value.</param>
        /// <returns>The animation progress from zero to one.</returns>
        protected override double ComputeProgress(
            PointF initValue,
            PointF targetValue,
            PointF currValue)
        {
            double progress = 1.0;
            progress = Math.Min(progress, ((double)currValue.X - (double)initValue.X) / ((double)targetValue.X - (double)initValue.X));
            progress = Math.Min(progress, ((double)currValue.Y - (double)initValue.Y) / ((double)targetValue.Y - (double)initValue.Y));

            return progress;
        }
    }

    #endregion
}
