﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/

#include <cassert>

#include <msclr/marshal.h>

#include "gfxTool_Main.h"
#include "ShaderConverterLogHandler.h"
#include "ShaderConverterWrapper.h"

using namespace std;
using namespace msclr::interop;

namespace EffectMaker {
namespace SpecGeneric {
namespace Shader {

//---------------------------------------------------------------------------
//  関数
//---------------------------------------------------------------------------

// 引数をパースしてC言語の argc, argv の形式に変換する
static void ParseArgs(String^ args, int* argc, char*** argv);

// argc, argv に変換した引数のデータを破棄する
static void ReleaseArgs(int argc, char** argv);

//---------------------------------------------------------------------------
//  コンストラクタです。
//---------------------------------------------------------------------------
ShaderConverterWrapper::ShaderConverterWrapper()
    : m_pShaderConverterDll()
{
}

//---------------------------------------------------------------------------
//  デストラクタです。
//---------------------------------------------------------------------------
ShaderConverterWrapper::~ShaderConverterWrapper()
{
    assert(m_pShaderConverterDll == nullptr);
}

//---------------------------------------------------------------------------
//  ファイナライザです。
//---------------------------------------------------------------------------
ShaderConverterWrapper::!ShaderConverterWrapper()
{
}

//---------------------------------------------------------------------------
//  DLLを初期化します。
//---------------------------------------------------------------------------
void ShaderConverterWrapper::Initialize(String^ dllPath)
{
    assert(m_pShaderConverterDll == nullptr);

    {
        marshal_context^ pContext = gcnew marshal_context();
        const char* pDllPath = pContext->marshal_as<const char*>(dllPath);

        m_pShaderConverterDll = new nn::gfxTool::ShaderConverterDll();
        m_pShaderConverterDll->Initialize(pDllPath);
    }
}

//---------------------------------------------------------------------------
//  DLLを破棄します。
//---------------------------------------------------------------------------
void ShaderConverterWrapper::Release()
{
    if (m_pShaderConverterDll != nullptr)
    {
        delete m_pShaderConverterDll;
        m_pShaderConverterDll = nullptr;
    }
}

//---------------------------------------------------------------------------
//  コンバート処理を実行します。
//---------------------------------------------------------------------------
ShaderConverterWrapper::ConvertResult^ ShaderConverterWrapper::Execute(String^ args)
{
    int argc;
    char** argv;

    // 引数を argc, argv に分解する
    ParseArgs(args, &argc, &argv);

    ConvertResult^ result = gcnew ConvertResult();

    // コンバート処理を実行する
    {
        ShaderConverterLogHandler logHandler;

        logHandler.Initialize(m_pShaderConverterDll);
        logHandler.ConnectPipe();

        result->ResultCode = nn::gfxTool::Execute(argc, argv, *m_pShaderConverterDll);

        logHandler.DisconnectPipe();

        result->DefaultLog = logHandler.ReadDefaultLogData();
        result->WarningLog = logHandler.ReadWarningLogData();
        result->ErrorLog   = logHandler.ReadErrorLogData();

        logHandler.Release();
    }

    // 引数データを破棄する
    ReleaseArgs(argc, argv);

    return result;
}

//---------------------------------------------------------------------------
//! @brief      引数をパースしてC言語の argc, argv の形式に変換します。
//! @param[in]  args  引数
//! @param[out] argc  argc
//! @param[out] argv  argv
//---------------------------------------------------------------------------
static void ParseArgs(String^ args, int* pArgc, char*** pArgv)
{
    int& argc = *pArgc;
    char**& argv = *pArgv;

    wchar_t** argv_w;

    // args を argv_w (wchar_t) に変換する
    {
        marshal_context^ pContext = gcnew marshal_context();
        const wchar_t* pArgs = pContext->marshal_as<const wchar_t*>(args);

        argv_w = CommandLineToArgvW(pArgs, &argc);
    }

    argv = new char*[argc];

    // argv_w (wchar_t) を argv (char) に変換する
    for (int i = 0; i < argc; ++i)
    {
        int length = WideCharToMultiByte(CP_ACP, 0, argv_w[i], -1, NULL, 0, NULL, NULL);

        if (length == 0)
        {
            argv[i] = new char[1] { '\0' };
            continue;
        }

        argv[i] = new char[length + 1];
        length = WideCharToMultiByte(CP_ACP, 0, argv_w[i], -1, argv[i], length, NULL, NULL);

        argv[i][length] = '\0';
    }

    LocalFree(argv_w);
}

//---------------------------------------------------------------------------
//! @brief     argc, argv に変換した引数のデータを破棄します。
//! @param[in] argc  argc
//! @param[in] argv  argv
//---------------------------------------------------------------------------
static void ReleaseArgs(int argc, char** argv)
{
    for (int i = 0; i < argc; ++i)
    {
        delete[] argv[i];
        argv[i] = nullptr;
    }

    delete[] argv;
}

}
}
}
