﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShaderConverterLib_ShaderConverter_H__
#define __ShaderConverterLib_ShaderConverter_H__

#pragma warning(disable:4251)

#include "ShaderConverterUtility.h"
#include <nw/eft/eft2_Binary.h>
#include <deque>

namespace EffectMaker {
namespace ShaderConverterLib {
    ////////////////////////////////////
    // ShaderConverter class
    ////////////////////////////////////
    class LIB_API ShaderConverter
    {
    public:
        //------------------------------------------------------------------------------
        //      コンストラクタ
        //------------------------------------------------------------------------------
        ShaderConverter(const char* szConverterFolderPath,
                        const char* szShaderFolderPath,
                        int         convertMode,
                        int         jobsNumber,
                        int         shaderAlignment,
                        int         binaryPosition,
                        int         IsSubBinaryConverting);

        //------------------------------------------------------------------------------
        //      デストラクタ
        //------------------------------------------------------------------------------
        ~ShaderConverter();

        //------------------------------------------------------------------------------
        //      初期化する
        //------------------------------------------------------------------------------
        void Initialize(ShaderCodeListCpp* pShaderCodes);

        //------------------------------------------------------------------------------
        //      シェーダバイナリを生成する
        //------------------------------------------------------------------------------
        u8* Convert( ShaderConverterEmitterData* pDataArray,
                     ShaderCompileErrorList*     pCompileErrorList,
                     s32                         dataCount,
                     s32*                        pBinarySize,
                     s32*                        pShaderCount );

        //------------------------------------------------------------------------------
        //      シェーダソースコードの生成
        //------------------------------------------------------------------------------
        bool GenerateShaderCode( ShaderConverterEmitterData* pEmitterData,
                                 char**                      ppVertexShader,
                                 s32*                        pVertexShaderSize,
                                 char**                      ppFragmentShader,
                                 s32*                        pFragmentShaderSize,
                                 char**                      ppComputeShader,
                                 s32*                        pComputeShaderSize,
                                 u32                         userDefineIndex = 0);

        //------------------------------------------------------------------------------
        //      シェーダアセンブリコードの生成
        //------------------------------------------------------------------------------
        bool GenerateAssemblyCode( ShaderConverterEmitterData* pEmitterData,
                                   ShaderCompileErrorList*     pCompileErrorList,
                                   char**                      ppVertexAssembly,
                                   s32*                        pVertexAssemblySize,
                                   char**                      ppFragmentAssembly,
                                   s32*                        pFragmentAssemblySize,
                                   char**                      ppComputeAssembly,
                                   s32*                        pComputeAssemblySize,
                                   u32                         userDefineIndex = 0);

private:
#if 1
        struct DataArrayInfo
        {
        public:
            DataArrayInfo() :
                shaderConverterEitterData(NULL),
                shaderIndex(-1),
                useShaderDef1Index(false),
                userShaderDef1Index(-1),
                useShaderDef2Index(false),
                userShaderDef2Index(-1)
            {
            }

            bool IsUserShaderDef1Enabled() { return useShaderDef1Index; }
            bool IsUserShaderDef2Enabled() { return useShaderDef2Index; }

            ShaderConverterEmitterData*  shaderConverterEitterData;
            s32                          shaderIndex;
            bool                         useShaderDef1Index;
            s32                          userShaderDef1Index;
            bool                         useShaderDef2Index;
            s32                          userShaderDef2Index;

        };
#endif
        //------------------------------------------------------------------------------
        //      エンディアンフリップをする
        //------------------------------------------------------------------------------
        u32 flip(u32 a, bool isBigEndian = true )
        {
            if (isBigEndian)
            {
                u32 tmpSize = a;
                nw::eft2::EndianUtil::Flip(&tmpSize);
                return tmpSize;
            }
            else
                return a;
        }

        //------------------------------------------------------------------------------
        //      エンディアンフリップをする(u16)
        //------------------------------------------------------------------------------
        u16 flip_u16(u16 a, bool isBigEndian = true )
        {
            if (isBigEndian)
            {
                u16 tmpSize = a;
                nw::eft2::EndianUtil::Flip(&tmpSize);
                return tmpSize;
            }
            else
                return a;
        }

        //------------------------------------------------------------------------------
        //      コンバートモード
        //------------------------------------------------------------------------------
        enum convertMode
        {
            CONVERT_FAST                = 0,
            CONVERT_DUP_BINARY,
            CONVERT_USE_SHADER_BINARY,
            CONVERT_BUILD_SHADER_BINARY
        };

        //------------------------------------------------------------------------------
        //      モード毎にシェーダを生成する
        //------------------------------------------------------------------------------
        u32 ConvertWithMode( ShaderConverterEmitterData* pEmitterData,
                             ShaderCompileErrorList*     pCompileErrorList,
                             nw::eft2::ResEmitter*       emitterData,
                             nw::eft2::ShaderKey*        shaderKey,
                             u32                         userDefIdx );

        //------------------------------------------------------------------------------
        //      頂点シェーダコードを生成する
        //------------------------------------------------------------------------------
        u32 GenerateVertexShaderCode( char* shaderCodeBuf,
                                      char* compileSettings,
                                      u32   settingSize,
                                      char* overrideVertexShader,
                                      s32   overrideVertexShaderLength,
                                      bool  useCustomShader,
                                      s32   customShaderIndex,
                                      s32   reservedShaderIndex,
                                      bool  useStreamOut = false );

        //------------------------------------------------------------------------------
        //      フラグメントシェーダコードを生成する
        //------------------------------------------------------------------------------
        u32 GenerateFragmentShaderCode( char* shaderCodeBuf,
                                        char* compileSettings,
                                        u32   settingSize,
                                        char* overrideFragmentSader,
                                        s32   overrideFragmentShaderLength,
                                        bool  useCustomShader,
                                        s32   customShaderIndex,
                                        s32   reservedShaderIndex,
                                        bool  useStreamOut = false );

        //------------------------------------------------------------------------------
        //      シェーダソースコードをコンパイルする
        //------------------------------------------------------------------------------
        s32  CompileShader( char* shaderVsh,
                            char* shaderFsh,
                            char* compileSettings,
                            u32   settingSize,
                            char* overrideShader,
                            s32   overrideShaderLength,
                            ShaderTable* shaderTbl,
                            bool  useCustomShader,
                            bool  useStreamOut = false );

        //------------------------------------------------------------------------------
        //      エミッタデータを取得する
        //------------------------------------------------------------------------------
        nw::eft2::ResEmitter* GetEmitter( u8 *pHeaderData, u8 *pEmitterData );

        //------------------------------------------------------------------------------
        //      エミッタプラグインデータを取得する
        //------------------------------------------------------------------------------
        void* GetEmitterPlugin( nw::eft2::BinaryData* binaryData );

        //------------------------------------------------------------------------------
        //      カスタムフィールドデータを取得する
        //------------------------------------------------------------------------------
        void* GetCustomField( nw::eft2::BinaryData* binaryData );

        //------------------------------------------------------------------------------
        //      フィールドデータの存在チェック
        //------------------------------------------------------------------------------
        bool IsFieldExist( nw::eft2::BinaryData* binaryData );

        //------------------------------------------------------------------------------
        //      シェーダキャッシュの使用をチェック
        //------------------------------------------------------------------------------
        bool IsUseShaderCache() { return ( ( mIsShaderCacheUse ) && ( mIsShaderCacheFileExist ) ) ? true : false; }

        //------------------------------------------------------------------------------
        //      シェーダキャッシュを出力するか？
        //------------------------------------------------------------------------------
        bool IsWriterShaderCache() { return ( ( mIsShaderCacheUse ) && ( mIsShaderCacheCreated ) ) ? true : false; }

        //------------------------------------------------------------------------------
        //     テキストエディタに表示する
        //------------------------------------------------------------------------------
        void OpenTextFile( char* outPath );

        //------------------------------------------------------------------------------
        // Output shader compile error message.
        //------------------------------------------------------------------------------
        void OutputShaderCompileErrorMessage( const char *szEmitterName, u32 userShaderDef = 0 );

        //------------------------------------------------------------------------------
        // Output error message when shader compile failed even without custom shader.
        //------------------------------------------------------------------------------
        void OutputShaderRecompileErrorMessage( const char *szEmitterName );

        //------------------------------------------------------------------------------
        // Output shader compile warning message.
        //------------------------------------------------------------------------------
        void OutputShaderCompileWarningMessage( const char *szEmitterName );

        //------------------------------------------------------------------------------
        // Output shader compile error log to debug console.
        //------------------------------------------------------------------------------
        void OutputShaderCompileErrorLog( const char *szEmitterName, const char *szErrorLog );

        //------------------------------------------------------------------------------
        // Output user generated msg to debug console.
        //------------------------------------------------------------------------------
        void OutputDebugMsg( const char *msgString );

        //------------------------------------------------------------------------------
        //      コンバート- 速さ重視　
        //------------------------------------------------------------------------------
        u32 ConvertFast(
            ShaderConverterEmitterData* pEmitterData,
            ShaderCompileErrorList*     pCompileErrorList,
            nw::eft2::ResEmitter*       emitterData,
            nw::eft2::ShaderKey*        shaderKey,
            u32                         userDefIdx );

        //------------------------------------------------------------------------------
        //      コンバート- シェーダバイナリを使用する　
        //------------------------------------------------------------------------------
        u32 ConvertShaderCache(
            ShaderConverterEmitterData* pEmitterData,
            ShaderCompileErrorList*     pCompileErrorList,
            nw::eft2::ResEmitter*       emitterData,
            nw::eft2::ShaderKey*        shaderKey );

        //------------------------------------------------------------------------------
        //      シェーダを生成する　
        //------------------------------------------------------------------------------
        s32 GenerateShader( ShaderConverterEmitterData* pEmitterData,
                            ShaderCompileErrorList *pCompileErrorList,
                            nw::eft2::ResEmitter* emitterData,
                            char* setting,
                            u32 settingSize,
                            ShaderTable* shaderTable,
                            u32 userShaderDef = 0 );

#if 1
        u32 ConvertInternal( ShaderConverterEmitterData *pDataArray,
                              ShaderCompileErrorList *pCompileErrorList,
                              s32 dataCount );

        void AddShaderDataArrayInfo( ShaderConverterEmitterData *pDataArray,
                                     s32 dataCount );

        void GenerateShaderCodeInternal( ShaderKeyInfo& shaderKeyInfo,
                                         bool useDefShaderKey = false );

        void AddShaderIndexToDataArray();
#endif

        const char*        mConverterFolderPath;         // shader compile dll directory
        const char*        mShaderFolderPath;            // default shader source directory
        char*              mShaderCachePath;

        ShaderCodeListCpp* mShaderCodes;

        ShaderCodeParam    mCustomVshSrc;
        ShaderCodeParam    mCustomFshSrc;

        char*              szVshCodeCompositeBuffer;
        char*              szFshCodeCompositeBuffer;

        ShaderKeyInfo*     mShaderKeyArray;
        ShaderTable*       mShaderTbl;
        u32                mShaderIdx;
        u32                mTotalShaderSize;

        bool               mIsField;                      // フィールドがあるかどうか？
        s32                mReservedShaderIndex;          // Reserved shader index. (-1 if not used)
        s32                mCustomShaderIndex;            // CustomShaderインデックス

        int                mConvertMode;
        bool               mIsCommandLineMode;

        bool               mIsShaderCacheUse;              // シェーダバイナリキャッシュを使用するかどうか？
        bool               mIsShaderCacheFileExist;        // シェーダバイナリファイルがあるかどうか？
        bool               mIsShaderCacheCreated;          // シェーダバイナリファイルを作成しました
        ShaderCache        mShaderCache;
        u32                mTmpShaderSourceCnt;
        nw::eft2::ShaderKey*
                           mDefaultShaderKey;

        std::deque<DataArrayInfo>
                           mDataArrayInfo;
        int                mJobsNumber;
        int                mBinaryPosition;
        int                mShaderAlignment;
        int                mIsSubBinaryConverting;
    };

} // namespace ShaderConverter
} // namespace EffectMaker


#endif // __ShaderConverterLib_ShaderConverter_H__
