﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShaderCodeGeneratorGeneric_PCH.h"
#include "ShaderCodeGeneratorGeneric.h"
#include "ShaderCodeListCs.h"

#include <nw/eft/eft2_Data.h>

#include <windows.h>
#include <cassert>
#include <string>

using namespace std;

using namespace System::Runtime::InteropServices;
using namespace System::Text;

namespace EffectMaker {
namespace ShaderCodeGeneratorGeneric {

    static char* CsStringToCppChar(String ^pStr, u32 *pSize);

    static std::string CompileCheck(GLuint type, const GLchar **source);

    /// <summary>
    /// Constructor
    /// </summary>
    ShaderCodeGeneratorGeneric::ShaderCodeGeneratorGeneric(System::IntPtr argHwnd) :
        m_pConverter(NULL),
        m_pShaderCodes(NULL),
        m_inputEmitterCount(0),
        m_pInputBuffer(NULL),
        m_hWnd((HWND)argHwnd.ToPointer()),
        m_hDC(nullptr),
        m_hRC(nullptr),
        bIsInitializedGLEW(false)
    {
        // Create the shader converter.
        m_pConverter = new ShaderConverter();
    }

    /// <summary>
    /// Finalize.
    /// </summary>
    ShaderCodeGeneratorGeneric::!ShaderCodeGeneratorGeneric()
    {
        if (m_hRC != nullptr)
        {
            if (wglGetCurrentContext() == m_hRC) wglMakeCurrent(nullptr, nullptr);
            wglDeleteContext(m_hRC);
            m_hRC = nullptr;
        }

        if (m_hDC != nullptr && m_hWnd != nullptr)
        {
            ReleaseDC(m_hWnd, m_hDC);
            m_hDC = nullptr;
            m_hWnd = nullptr;
        }

        // Release the converter.
        if (this->m_pConverter != NULL)
        {
            delete this->m_pConverter;
            this->m_pConverter = NULL;
        }

        // Release the buffers.
        this->ReleaseInputDataBuffer();

        // Release the shader codes.
        if (this->m_pShaderCodes != NULL)
        {
            if (this->m_pShaderCodes->mSpecDecShaderSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mSpecDecShaderSrc.buf;
                this->m_pShaderCodes->mSpecDecShaderSrc.buf = NULL;
                this->m_pShaderCodes->mSpecDecShaderSrc.size = 0;
            }

            if (this->m_pShaderCodes->mVertexShaderSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mVertexShaderSrc.buf;
                this->m_pShaderCodes->mVertexShaderSrc.buf = NULL;
                this->m_pShaderCodes->mVertexShaderSrc.size = 0;
            }

            if (this->m_pShaderCodes->mFragShaderSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mFragShaderSrc.buf;
                this->m_pShaderCodes->mFragShaderSrc.buf = NULL;
                this->m_pShaderCodes->mFragShaderSrc.size = 0;
            }

            if (this->m_pShaderCodes->mParticleDecVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mParticleDecVshSrc.buf;
                this->m_pShaderCodes->mParticleDecVshSrc.buf = NULL;
                this->m_pShaderCodes->mParticleDecVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mParticleDecFshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mParticleDecFshSrc.buf;
                this->m_pShaderCodes->mParticleDecFshSrc.buf = NULL;
                this->m_pShaderCodes->mParticleDecFshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mStreamOutDecVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mStreamOutDecVshSrc.buf;
                this->m_pShaderCodes->mStreamOutDecVshSrc.buf = NULL;
                this->m_pShaderCodes->mStreamOutDecVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mStreamOutVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mStreamOutVshSrc.buf;
                this->m_pShaderCodes->mStreamOutVshSrc.buf = NULL;
                this->m_pShaderCodes->mStreamOutVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mGeneralVshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mGeneralVshSrc.buf;
                this->m_pShaderCodes->mGeneralVshSrc.buf = NULL;
                this->m_pShaderCodes->mGeneralVshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mGeneralFshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mGeneralFshSrc.buf;
                this->m_pShaderCodes->mGeneralFshSrc.buf = NULL;
                this->m_pShaderCodes->mGeneralFshSrc.size = 0;
            }

            if (this->m_pShaderCodes->mGeneralCshSrc.buf != NULL)
            {
                delete[] this->m_pShaderCodes->mGeneralCshSrc.buf;
                this->m_pShaderCodes->mGeneralCshSrc.buf = NULL;
                this->m_pShaderCodes->mGeneralCshSrc.size = 0;
            }

            for (int i = 0; i < 8; ++i)
            {
                if (this->m_pShaderCodes->mCustomVshSrc[i].buf != NULL)
                {
                    delete[] this->m_pShaderCodes->mCustomVshSrc[i].buf;
                    this->m_pShaderCodes->mCustomVshSrc[i].buf = NULL;
                    this->m_pShaderCodes->mCustomVshSrc[i].size = 0;
                }

                if (this->m_pShaderCodes->mCustomFshSrc[i].buf != NULL)
                {
                    delete[] this->m_pShaderCodes->mCustomFshSrc[i].buf;
                    this->m_pShaderCodes->mCustomFshSrc[i].buf = NULL;
                    this->m_pShaderCodes->mCustomFshSrc[i].size = 0;
                }
            }

            delete m_pShaderCodes;
            m_pShaderCodes = NULL;
        }
    }

    /// <summary>
    /// Destructor.
    /// </summary>
    ShaderCodeGeneratorGeneric::~ShaderCodeGeneratorGeneric()
    {
        this->!ShaderCodeGeneratorGeneric();
    }

    /// <summary>
    /// シェーダコードを設定します.
    /// </summary>
    /// <param name="pShaderCodes">シェーダコードです.</param>
    void ShaderCodeGeneratorGeneric::SetShaderCodes(ShaderCodeListCs^ pShaderCodes)
    {
        assert(m_pShaderCodes == NULL);

        m_pShaderCodes = new ShaderCodeListCpp();

        if (pShaderCodes->SpecDecSrc != nullptr)
        {
            this->m_pShaderCodes->mSpecDecShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->SpecDecSrc,
                (u32*)&this->m_pShaderCodes->mSpecDecShaderSrc.size);
        }

        if (pShaderCodes->VertexShaderSrc != nullptr)
        {
            this->m_pShaderCodes->mVertexShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->VertexShaderSrc,
                (u32*)&this->m_pShaderCodes->mVertexShaderSrc.size);
        }

        if (pShaderCodes->ParticleGlslSrc != nullptr)
        {
            this->m_pShaderCodes->mParticleGlslSrc.buf = CsStringToCppChar(
                pShaderCodes->ParticleGlslSrc,
                (u32*)&this->m_pShaderCodes->mParticleGlslSrc.size);
        }

        if (pShaderCodes->FragShaderSrc != nullptr)
        {
            this->m_pShaderCodes->mFragShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->FragShaderSrc,
                (u32*)&this->m_pShaderCodes->mFragShaderSrc.size);
        }

        if (pShaderCodes->ParticleDecVshSrc != nullptr)
        {
            this->m_pShaderCodes->mParticleDecVshSrc.buf = CsStringToCppChar(
                pShaderCodes->ParticleDecVshSrc,
                (u32*)&this->m_pShaderCodes->mParticleDecVshSrc.size);
        }

        if (pShaderCodes->ParticleDecFshSrc != nullptr)
        {
            this->m_pShaderCodes->mParticleDecFshSrc.buf = CsStringToCppChar(
                pShaderCodes->ParticleDecFshSrc,
                (u32*)&this->m_pShaderCodes->mParticleDecFshSrc.size);
        }

        if (pShaderCodes->StripShaderSrc != nullptr)
        {
            this->m_pShaderCodes->mStripShaderSrc.buf = CsStringToCppChar(
                pShaderCodes->StripShaderSrc,
                (u32*)&this->m_pShaderCodes->mStripShaderSrc.size);
        }

        if (pShaderCodes->GeneralVshSrc != nullptr)
        {
            this->m_pShaderCodes->mGeneralVshSrc.buf = CsStringToCppChar(
                pShaderCodes->GeneralVshSrc,
                (u32*)&this->m_pShaderCodes->mGeneralVshSrc.size);
        }

        if (pShaderCodes->GeneralFshSrc != nullptr)
        {
            this->m_pShaderCodes->mGeneralFshSrc.buf = CsStringToCppChar(
                pShaderCodes->GeneralFshSrc,
                (u32*)&this->m_pShaderCodes->mGeneralFshSrc.size);
        }

        if (pShaderCodes->GeneralCshSrc != nullptr)
        {
            this->m_pShaderCodes->mGeneralCshSrc.buf = CsStringToCppChar(
                pShaderCodes->GeneralCshSrc,
                (u32*)&this->m_pShaderCodes->mGeneralCshSrc.size);
        }

        if (pShaderCodes->StreamOutDecVshSrc != nullptr)
        {
            this->m_pShaderCodes->mStreamOutDecVshSrc.buf = CsStringToCppChar(
                pShaderCodes->StreamOutDecVshSrc,
                (u32*)&this->m_pShaderCodes->mStreamOutDecVshSrc.size);
        }

        if (pShaderCodes->StreamOutVshSrc != nullptr)
        {
            this->m_pShaderCodes->mStreamOutVshSrc.buf = CsStringToCppChar(
                pShaderCodes->StreamOutVshSrc,
                (u32*)&this->m_pShaderCodes->mStreamOutVshSrc.size);
        }

        for (int i = 0; i < 8; ++i)
        {
            if (pShaderCodes->CustomVshSrc != nullptr &&
                pShaderCodes->CustomVshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mCustomVshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->CustomVshSrc[i],
                    (u32*)&this->m_pShaderCodes->mCustomVshSrc[i].size);
            }

            if (pShaderCodes->CustomFshSrc != nullptr &&
                pShaderCodes->CustomFshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mCustomFshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->CustomFshSrc[i],
                    (u32*)&this->m_pShaderCodes->mCustomFshSrc[i].size);
            }
        }

        for (int i = 0; i < 8; ++i)
        {
            if (pShaderCodes->ReservedVshSrc != nullptr &&
                pShaderCodes->ReservedVshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mReservedVshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->ReservedVshSrc[i],
                    (u32*)&this->m_pShaderCodes->mReservedVshSrc[i].size);
            }

            if (pShaderCodes->ReservedFshSrc != nullptr &&
                pShaderCodes->ReservedFshSrc[i] != nullptr)
            {
                this->m_pShaderCodes->mReservedFshSrc[i].buf = CsStringToCppChar(
                    pShaderCodes->ReservedFshSrc[i],
                    (u32*)&this->m_pShaderCodes->mReservedFshSrc[i].size);
            }
        }
    }

    /// <summary>
    /// Generate shader code for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader code.</param>
    /// <param name="pFragmentShader">The output fragment shader code.</param>
    /// <param name="pComputeShader">The output compute shader code.</param>
    /// <returns>True on success.</returns>
    bool ShaderCodeGeneratorGeneric::GenerateShaderCode(ShaderConversionInputData ^pInputData,
                                                   [Out] System::String^% pVertexShader,
                                                   [Out] System::String^% pFragmentShader,
                                                   [Out] System::String^% pComputeShader)
    {
        pVertexShader   = nullptr;
        pFragmentShader = nullptr;
        pComputeShader  = nullptr;

        if (this->m_pConverter == NULL)
        {
            return false;
        }

        this->ConvertInputData(pInputData);
        if (this->m_pInputBuffer == NULL)
        {
            return false;
        }

        if (this->m_pShaderCodes == NULL)
        {
            return false;
        }

        this->m_pConverter->Initialize(m_pShaderCodes);

        int vertexShaderSize = 0;
        int fragmentShaderSize = 0;
        int computeShaderSize = 0;
        char *pNativeVertexShader = NULL;
        char *pNativeFragmentShader = NULL;
        char *pNativeComputeShader = NULL;

        bool result = this->m_pConverter->GenerateShaderCode(
            (ShaderConverterEmitterData*)this->m_pInputBuffer,
            &pNativeVertexShader,
            &vertexShaderSize,
            &pNativeFragmentShader,
            &fragmentShaderSize,
            &pNativeComputeShader,
            &computeShaderSize,
            pInputData->UserDefineIndex);

        // Release the buffer.
        this->ReleaseInputDataBuffer();

        if (result == false)
        {
            pVertexShader = nullptr;
            pFragmentShader = nullptr;
            pComputeShader = nullptr;
            return false;
        }

        Encoding ^pEncoding = Encoding::GetEncoding("Shift_JIS");

        pVertexShader = gcnew String(pNativeVertexShader, 0, vertexShaderSize, pEncoding);
        pFragmentShader = gcnew String(pNativeFragmentShader, 0, fragmentShaderSize, pEncoding);
        pComputeShader = gcnew String(pNativeComputeShader, 0, computeShaderSize, pEncoding);

        // Clean up.
        delete[] pNativeVertexShader;
        delete[] pNativeFragmentShader;
        delete[] pNativeComputeShader;

        return result;
    }

    /// <summary>
    /// Generate shader assembly for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader assembly.</param>
    /// <param name="pFragmentShader">The output fragment shader assembly.</param>
    /// <param name="pComputeShader">The output compute shader assembly.</param>
    /// <param name="pErrorList">The array of compile error information.</param>
    /// <returns>True on success.</returns>
    bool ShaderCodeGeneratorGeneric::GenerateShaderAssembly(ShaderConversionInputData ^pInputData,
                                                            [Out] System::String^% pVertexShader,
                                                            [Out] System::String^% pFragmentShader,
                                                            [Out] System::String^% pComputeShader,
                                                            [Out] List<ShaderCompileErrorInfo^>^% pErrorList)
    {
        pVertexShader   = nullptr;
        pFragmentShader = nullptr;
        pComputeShader  = nullptr;
        pErrorList      = gcnew List<ShaderCompileErrorInfo^>();

        if (this->m_pConverter == NULL)
        {
            return false;
        }

        this->ConvertInputData(pInputData);
        if (this->m_pInputBuffer == NULL)
        {
            return false;
        }

        if (this->m_pShaderCodes == NULL)
        {
            return false;
        }

        this->m_pConverter->Initialize(m_pShaderCodes);

        int vertexShaderSize = 0;
        int fragmentShaderSize = 0;
        int computeShaderSize = 0;
        char *pNativeVertexShader = NULL;
        char *pNativeFragmentShader = NULL;
        char *pNativeComputeShader = NULL;
        ShaderCompileErrorList errorList;

        // エミッタ名を表示するなら次のようにしてResEmitterを取得する必要がある
        ShaderConverterEmitterData *sced = (ShaderConverterEmitterData*)this->m_pInputBuffer;
        nw::eft2::BinaryData* header = reinterpret_cast<nw::eft2::BinaryData*>(sced->pBinaryHeader);
        nw::eft2::ResEmitter* resEmitter = reinterpret_cast<nw::eft2::ResEmitter*>(header->GetBinaryData());

        bool result = this->m_pConverter->GenerateShaderCode(
            (ShaderConverterEmitterData*)this->m_pInputBuffer,
            &pNativeVertexShader,
            &vertexShaderSize,
            &pNativeFragmentShader,
            &fragmentShaderSize,
            &pNativeComputeShader,
            &computeShaderSize,
            pInputData->UserDefineIndex);

        std::string vsResult = CompileCheck(GL_VERTEX_SHADER, (const GLchar**)&pNativeVertexShader);
        if(!vsResult.empty())
        {
            errorList.AddErrorLog(resEmitter->name, pNativeVertexShader, pNativeFragmentShader, vsResult.c_str());
        }

        std::string fsResult = CompileCheck(GL_FRAGMENT_SHADER, (const GLchar**)&pNativeFragmentShader);
        if(!fsResult.empty())
        {
            errorList.AddErrorLog(resEmitter->name, pNativeVertexShader, pNativeFragmentShader, fsResult.c_str());
        }

        // Release the buffer.
        this->ReleaseInputDataBuffer();

        bool isWarning;

        const char *szErrorEmitterName    = NULL;
        const char *szErrorVertexShader   = NULL;
        const char *szErrorFragmentShader = NULL;
        const char *szErrorLog            = NULL;

        // The compile error information.
        errorList.BeginEnumeration();
        while (errorList.GetNextItem(&isWarning,
                                     szErrorEmitterName,
                                     szErrorVertexShader,
                                     szErrorFragmentShader,
                                     szErrorLog) == true)
        {
            pErrorList->Add(gcnew ShaderCompileErrorInfo(isWarning,
                                                         szErrorEmitterName,
                                                         szErrorVertexShader,
                                                         szErrorFragmentShader,
                                                         szErrorLog));
        }

        if (result == false)
        {
            pVertexShader = nullptr;
            pFragmentShader = nullptr;
            pComputeShader = nullptr;
            return false;
        }

        Encoding ^pEncoding = Encoding::GetEncoding("Shift_JIS");

        pVertexShader = gcnew String(pNativeVertexShader, 0, vertexShaderSize, pEncoding);
        pFragmentShader = gcnew String(pNativeFragmentShader, 0, fragmentShaderSize, pEncoding);
        pComputeShader = gcnew String(pNativeComputeShader, 0, computeShaderSize, pEncoding);

        // Clean up.
        delete[] pNativeVertexShader;
        delete[] pNativeFragmentShader;
        delete[] pNativeComputeShader;

        return result;
    }

    /// <summary>
    /// Generate vfx shader key and codes for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pShaderKey">The output shader key code.</param>
    /// <param name="pVertexShader">The output vertex shader code.</param>
    /// <param name="pFragmentShader">The output fragment shader code.</param>
    /// <returns>True on success.</returns>
    bool ShaderCodeGeneratorGeneric::GenerateVfxShaderKeyAndCodes(ShaderConversionInputData ^pInputData,
                                                                  [Out] System::String^% pShaderKey,
                                                                  [Out] System::String^% pVertexShader,
                                                                  [Out] System::String^% pFragmentShader,
                                                                  [Out] System::String^% pComputeShader)
    {
        pVertexShader   = nullptr;
        pFragmentShader = nullptr;

        if (this->m_pConverter == NULL)
        {
            return false;
        }

        this->ConvertInputData(pInputData);
        if (this->m_pInputBuffer == NULL)
        {
            return false;
        }

        if (this->m_pShaderCodes == NULL)
        {
            return false;
        }

        this->m_pConverter->Initialize(m_pShaderCodes);

        int shaderKeySize = 0;
        int vertexShaderSize = 0;
        int fragmentShaderSize = 0;
        int computeShaderSize = 0;
        char *pNativeShaderKey = NULL;
        char *pNativeVertexShader = NULL;
        char *pNativeFragmentShader = NULL;
        char *pNativeComputeShader = NULL;

        bool result = this->m_pConverter->GenerateVfxShaderKeyAndCodes(
            (ShaderConverterEmitterData*)this->m_pInputBuffer,
            &pNativeShaderKey,
            &shaderKeySize,
            &pNativeVertexShader,
            &vertexShaderSize,
            &pNativeFragmentShader,
            &fragmentShaderSize,
            &pNativeComputeShader,
            &computeShaderSize,
            pInputData->UserDefineIndex);

        ShaderConverterEmitterData *pEmitterDataArray =
            (ShaderConverterEmitterData*)this->m_pInputBuffer;

        pInputData->ShaderIndexOffset =
            (System::Int64)pEmitterDataArray->shaderIndexOffset;

        pInputData->VertexShaderIndex      = pEmitterDataArray->vertexShaderIndex;
        pInputData->PixelShaderIndex       = pEmitterDataArray->pixelShaderIndex;
        pInputData->UserVertexShaderIndex1 = pEmitterDataArray->userVertexShaderIndex1;
        pInputData->UserPixelShaderIndex1  = pEmitterDataArray->userPixelShaderIndex1;
        pInputData->UserVertexShaderIndex2 = pEmitterDataArray->userVertexShaderIndex2;
        pInputData->UserPixelShaderIndex2  = pEmitterDataArray->userPixelShaderIndex2;

        // Release the buffer.
        this->ReleaseInputDataBuffer();

        if (result == false)
        {
            pShaderKey = nullptr;
            pVertexShader = nullptr;
            pFragmentShader = nullptr;
            return false;
        }

        Encoding ^pEncoding = Encoding::GetEncoding("Shift_JIS");

        pShaderKey = gcnew String(pNativeShaderKey, 0, shaderKeySize, pEncoding);
        pVertexShader = gcnew String(pNativeVertexShader, 0, vertexShaderSize, pEncoding);
        pFragmentShader = gcnew String(pNativeFragmentShader, 0, fragmentShaderSize, pEncoding);
        pComputeShader = gcnew String(pNativeComputeShader, 0, computeShaderSize, pEncoding);

        // Clean up.
        delete[] pNativeShaderKey;
        delete[] pNativeVertexShader;
        delete[] pNativeFragmentShader;
        delete[] pNativeComputeShader;

        return true;
    }

    /// <summary>
    /// OpenGLとGLEWを初期化し、シェーダコンパイラを有効にします。
    /// </summary>
    bool ShaderCodeGeneratorGeneric::InitializeGLEW()
    {
        // OpenGL の初期化
        GLuint PixelFormat;
        PIXELFORMATDESCRIPTOR pfd;
        memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));

        pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
        pfd.nVersion = 1;
        pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
        pfd.iPixelType = PFD_TYPE_RGBA;
        pfd.cColorBits = 32;
        pfd.cDepthBits = 32;

        m_hDC = GetDC(m_hWnd);
        PixelFormat = ChoosePixelFormat(m_hDC, &pfd);
        if (SetPixelFormat(m_hDC, PixelFormat, &pfd) == FALSE)
        {
            ReleaseDC(m_hWnd, m_hDC);
            m_hDC = nullptr;
            return false;
        }

        m_hRC = wglCreateContext(m_hDC);
        if (m_hRC == nullptr)
        {
            ReleaseDC(m_hWnd, m_hDC);
            m_hDC = nullptr;
            return false;
        }

        wglMakeCurrent(m_hDC, m_hRC);

        glewInit();

        if ( glCreateShader == NULL || glShaderSource == NULL || glCompileShader == NULL )
        {
            return false;
        }

        return true;
    }

    /// <summary>
    /// Convert the input data into the native input data buffer
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    void ShaderCodeGeneratorGeneric::ConvertInputData(ShaderConversionInputData ^pInputData)
    {
        // Release the buffer first.
        this->ReleaseInputDataBuffer();

        // Save the emitter count.
        this->m_inputEmitterCount = 1;

        // Compute the required size for the input data.
        size_t emitterDataBlockSize = sizeof(ShaderConverterEmitterData);
        size_t requiredBufferSize = emitterDataBlockSize + pInputData->Size;

        // Create a native buffer for input data, in the following format :
        // [ShaderConverterEmitterData_0] <-- Emitter data block
        // [ShaderConverterEmitterData_1]
        // ...
        // [ShaderConverterEmitterData_N]
        // [Emitter binary, override shader code] <-- Raw data block
        // [Emitter binary, override shader code]
        // ...
        // [Emitter binary, override shader code]
        this->m_pInputBuffer = new unsigned char[requiredBufferSize];

        // The pointers.
        ShaderConverterEmitterData *pEmitterData =
            (ShaderConverterEmitterData*)this->m_pInputBuffer;

        unsigned char *pRawData =
            this->m_pInputBuffer + emitterDataBlockSize;

        // Fill in the data.
        this->SetupNativeInputData(pInputData, &pEmitterData, &pRawData);
    }

    /// <summary>
    /// Convert and copy the managed input data into the native memory,
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pManagedData">The managed input data.</param>
    /// <param name="ppEmitterData">The pointer to the native emitter data.</param>
    /// <param name="ppRawData">The pointer to the native emitter binary and custom shaders.</param>
    void ShaderCodeGeneratorGeneric::SetupNativeInputData(ShaderConversionInputData ^pManagedData,
                                                     ShaderConverterEmitterData **ppEmitterData,
                                                     unsigned char **ppRawData)
    {
        // Set up the emitter data.
        (*ppEmitterData)->pBinaryHeader = *ppRawData;

        // ヘッダからのオフセットを取得して加算することでResEmitterの正しいアドレスを求める
        pin_ptr<Byte> pBinHeader = &(pManagedData->BinaryHeader[0]);
        nw::eft2::BinaryData *binHeader = (nw::eft2::BinaryData *)pBinHeader;
        (*ppEmitterData)->pResEmitter = (u8*)( *ppRawData + binHeader->offset );

        (*ppEmitterData)->szOverrideShader =
            (char*)(*ppRawData + pManagedData->BinaryHeader->Length + pManagedData->OverrideShaderOffset);

        (*ppEmitterData)->iOverrideShaderLength = Encoding::GetEncoding("Shift_JIS")->GetByteCount(pManagedData->OverrideShaderCode);
        (*ppEmitterData)->vertexShaderIndex    = 0;
        (*ppEmitterData)->pixelShaderIndex     = 0;
        (*ppEmitterData)->shaderIndexOffset    = 0;
        (*ppEmitterData)->reservedShaderIndex  = pManagedData->ReservedShaderIndex;

        // Set up the raw data.
        // The emitter binary.
        pin_ptr<unsigned char> pUnmanagedHeader = &(pManagedData->BinaryHeader[0]);
        pin_ptr<unsigned char> pUnmanagedBuffer = &(pManagedData->EmitterBinary[0]);

        size_t copySize = pManagedData->BinaryHeader->Length;
        memcpy_s(*ppRawData,
                 copySize,
                 pUnmanagedHeader,
                 copySize);

        copySize = pManagedData->EmitterBinary->Length;
        memcpy_s(*ppRawData + pManagedData->BinaryHeader->Length,
                 copySize,
                 pUnmanagedBuffer,
                 copySize);

        // The vertex shader.
        IntPtr szOverrideShader = Marshal::StringToHGlobalAnsi(pManagedData->OverrideShaderCode);

        copySize = System::Text::Encoding::GetEncoding("Shift_JIS")->GetByteCount(pManagedData->OverrideShaderCode) + 1;
        memcpy_s(*ppRawData + pManagedData->BinaryHeader->Length + pManagedData->OverrideShaderOffset,
                 copySize,
                 (const char*)szOverrideShader.ToPointer(),
                 copySize);

        Marshal::FreeHGlobal(szOverrideShader);

        // Advance the pointer.
        ++(*ppEmitterData);
        (*ppRawData) += pManagedData->Size;
    }

    /// <summary>
    /// Release the memory allocated for the native input data.
    /// </summary>
    void ShaderCodeGeneratorGeneric::ReleaseInputDataBuffer()
    {
        if (this->m_pInputBuffer != NULL)
        {
            delete[] this->m_pInputBuffer;
            this->m_pInputBuffer = NULL;
        }
    }

    /// <summary>
    /// C#のStringをC++のchar*に変換します.
    /// この関数で取得したchar*は不要になったらdelete[]してください.
    /// </summary>
    /// <param name="str">C#のString</param>
    /// <return>CPPのchar*を返します.</return>
    static char* CsStringToCppChar(String ^pStr, u32 *pSize)
    {
        int length = Encoding::GetEncoding("Shift_JIS")->GetByteCount(pStr);
        IntPtr strPtr = Marshal::StringToHGlobalAnsi(pStr);

        char *pChar = new char[length + 1];
        strcpy_s(
            pChar,
            length + 1,
            (const char*)strPtr.ToPointer());

        Marshal::FreeHGlobal(strPtr);

        if (pSize != NULL)
        {
            *pSize = length;
        }

        return pChar;
    }

    /// <summary>
    /// シェーダコードをコンパイルしてエラーログを取得します。
    /// </summary>
    /// <param name="type">シェーダのタイプ(GL_VERTEX_SHADER / GL_FRAGMENT_SHADER)</param>
    /// <param name="source">シェーダコードへのダブルポインタ</param>
    /// <return>エラーがあればログ文字列、なければ空文字列.</return>
    std::string ShaderCodeGeneratorGeneric::CompileCheck(GLuint type, const GLchar **source)
    {
        if ( bIsInitializedGLEW == false )
        {
            bIsInitializedGLEW = InitializeGLEW();
            assert(bIsInitializedGLEW);
        }

        std::string infoLog;
        GLsizei bufSize;
        GLuint id = glCreateShader(type);

        glShaderSource(id, 1, source, nullptr);
        glCompileShader(id);
        glGetShaderiv(id, GL_INFO_LOG_LENGTH , &bufSize);
        if(bufSize > 1)
        {
            GLchar *tmpBuf = (GLchar *)malloc(bufSize);

            if(tmpBuf != nullptr)
            {
                GLsizei length;
                glGetShaderInfoLog(id, bufSize, &length, tmpBuf);
                infoLog = tmpBuf;
                free(tmpBuf);
            }
        }

        glDeleteShader(id);

        return infoLog;
    }

} // namespace ShaderCodeGeneratorGeneric
} // namespace EffectMaker
