﻿
// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Diagnostics;
using System.Threading;
using EffectMaker.ProcessBridge;

namespace EffectMaker.Application
{
    /// <summary>
    /// 通信コマンドを処理するクラスです。
    /// </summary>
    public class BridgeCommandProcessor
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="mainForm">メインフォーム</param>
        /// <param name="syncContext">同期オブジェクト</param>
        public BridgeCommandProcessor(MainForm mainForm, SynchronizationContext syncContext)
        {
            this.mainForm = mainForm;
            this.syncContext = syncContext;
        }

        /// <summary>
        /// ファイナライザです。
        /// </summary>
        ~BridgeCommandProcessor()
        {
            Debug.Assert(this.bridgeServer == null, "未破棄");
        }

        /// <summary>
        /// 通信サーバーを初期化します。
        /// </summary>
        /// <returns></returns>
        public bool Initialize()
        {
            Debug.Assert(this.bridgeServer == null, "二重初期化");

            this.bridgeServer = new BridgeServer();
            bool resInit = this.bridgeServer.Initialize();

            if (resInit == false)
            {
                this.bridgeServer = null;

                this.Release();
                return false;
            }

            this.bridgeServer.Process += this.OnProcess;

            return true;
        }

        /// <summary>
        /// 通信サーバーを破棄します。
        /// </summary>
        public void Release()
        {
            if (this.bridgeServer != null)
            {
                this.bridgeServer.Process -= this.OnProcess;

                this.bridgeServer.Release();
                this.bridgeServer = null;
            }
        }

        /// <summary>
        /// 通信コマンドを処理します。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="remoteObject">リモートオブジェクト</param>
        private void OnProcess(object sender, BridgeRemoteObject remoteObject)
        {
            if (remoteObject.CommandType == BridgeCommandType.LoadFiles)
            {
                this.syncContext.Send(s => { this.mainForm.LoadFiles(remoteObject.Paths); }, null);
            }
        }

        /// <summary>
        /// メインフォーム
        /// </summary>
        private MainForm mainForm;

        /// <summary>
        /// 同期オブジェクト
        /// </summary>
        private SynchronizationContext syncContext;

        /// <summary>
        /// 通信サーバー
        /// </summary>
        private BridgeServer bridgeServer;
    }
}
