﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.Foundation.EventArguments;
using EffectMaker.UIControls;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Input;
using EffectMaker.UIControls.Specifics.TabPages;
using EffectMaker.UIControls.Xaml;
using EffectMaker.UILogic.Manager;
using EffectMaker.UILogic.ViewModels;
using WeifenLuo.WinFormsUI.Docking;

namespace EffectMaker.Application
{
    /// <summary>
    /// プロパティダイアログ
    /// </summary>
    public partial class PropertyDialog : WeifenLuo.WinFormsUI.Docking.DockContent
    {
        /// <summary>
        /// ビューモデルのルートです。
        /// </summary>
        private WorkspaceRootViewModel rootViewModel;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="rootViewModel">ビューモデルのルート</param>
        /// <param name="executableHolder">ExecutableHolder</param>
        public PropertyDialog(WorkspaceRootViewModel rootViewModel, MainFormExecutableHolder executableHolder)
        {
            this.InitializeComponent();

            // ビューモデルを設定
            this.rootViewModel = rootViewModel;
            this.workspaceControl.RootViewModel = this.rootViewModel;

            // ExecutableHolderを設定
            this.workspaceControl.ExecutableHolder = executableHolder;

            // プロパティパネルを作成
            this.PropertyPanel = new PropertyPanel();
            this.spcWorkspaceAndPropertyTab.Panel2.Controls.Add(this.PropertyPanel);

            this.workspaceControl.SelectedNodeChanged += this.OnWorkspaceSelectedNodeChanged;

            var selector = new PropertyPageItemContainerSelector();
            this.PropertyPanel.ItemContainerSelector = selector;
            this.PropertyPanel.Dock = DockStyle.Fill;
            this.PropertyPanel.Visibility = EffectMaker.UIControls.Layout.Visibility.Visible;

            selector.DataSourceModifiedChanged += (sender, args) => this.PropertyPanel.Invalidate();

            // ビルドインページのプリロード
            // 動作に問題はなさそうだが、起動時の3秒待ちとタブ初回表示時のつっかかり、
            // どちらを取るかと言われたら後者のほうが正直マシな感がある。
            ////selector.PreLoadPages();

            // PlaybackControllのイベントを登録
            WorkspaceController controller = this.rootViewModel.Controller;

            this.playbackControl.BackButtonClicked += (ss, ee) => controller.PlaybackBack();
            this.playbackControl.StopButtonClicked += (ss, ee) => controller.PlaybackStop();
            this.playbackControl.PlayButtonClicked += (ss, ee) => controller.PlaybackPlay();
            this.playbackControl.StepButtonClicked += (ss, ee) => controller.PlaybackStep();
            this.playbackControl.FadeButtonClicked += (ss, ee) => controller.PlaybackFade();
            this.playbackControl.LoopButtonClicked += (ss, ee) => controller.PlaybackLoop(ee.IsLoop, ee.Frame);
            this.playbackControl.LoopFrameChanged += (ss, ee) => controller.PlaybackLoop(ee.IsLoop, ee.Frame);

            // フォームを閉じる前に呼び出されるイベント.
            this.FormClosing += (s, e) =>
            {
                // 閉じないように、キャンセルする.
                e.Cancel = true;

                // 非表示にする.
                this.Hide();
            };
        }

        /// <summary>
        /// Gets the property panel instance.
        /// </summary>
        public PropertyPanel PropertyPanel { get; private set; }

        /// <summary>
        /// ワークスペース上の選択ノードが変更された場合に、追加で実行されるデリゲート
        /// </summary>
        public Action OnWorkspaceSelectedNodeChangedAdditionalAction { get; set; }

        /// <summary>
        /// 名前の変更を行います.
        /// </summary>
        public void Rename()
        {
            this.workspaceControl.Rename();
        }

        /// <summary>
        /// The detach selected node changed.
        /// </summary>
        public void DetachSelectedNodeChanged()
        {
            this.workspaceControl.SelectedNodeChanged -= this.OnWorkspaceSelectedNodeChanged;
            this.workspaceControl.ClearDataContext();
        }

        /// <summary>
        /// 現在表示しているページのヘルプリンク情報を取得します。
        /// </summary>
        /// <returns>ヘルプリンクを含むコントロール(ヘルプがない場合はnull).</returns>
        public IDocumentLinker GetCurrentPageLink()
        {
            int index = this.PropertyPanel.SelectedIndex;
            if (0 > index || index >= this.PropertyPanel.TabPages.Count)
            {
                return null;
            }

            return this.PropertyPanel.TabPages[index].Controls.OfType<IDocumentLinker>()
                       .FirstOrDefault(d => !string.IsNullOrEmpty(d.DocumentId));
        }

        /// <summary>
        /// フローティングウィンドウ時のショートカットキーハンドリング
        /// </summary>
        /// <param name="msg">メッセージ</param>
        /// <param name="keyData">キーデータ</param>
        /// <returns>処理済みならtrueを返す。</returns>
        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            if (this.DockState == DockState.Float)
            {
                GlobalKeyEventHandler.Instance.ProcessShortcut(keyData);
            }

            return base.ProcessCmdKey(ref msg, keyData);
        }

        /// <summary>
        /// Called when a node is selected or unselected on the Workspace tree view.
        /// </summary>
        /// <param name="sender">Event caller.</param>
        /// <param name="e">Event argument.</param>
        private void OnWorkspaceSelectedNodeChanged(
            object sender,
            ValueChangedEventArgs<TreeNode> e)
        {
            object dataItem = ControlExtensions.GetDataItemFromContainer(e.NewValue as IControl);
            this.PropertyPanel.DataContext = dataItem;

            // デリゲートが設定されていたら、実行する.
            if (this.OnWorkspaceSelectedNodeChangedAdditionalAction != null)
            {
                this.OnWorkspaceSelectedNodeChangedAdditionalAction();
            }
        }
    }
}
