﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.Serializer;
using EffectMaker.DataModel.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.UILogic.ViewModels.IO;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// Command line processor for updating out-dated emitter set files.
    /// </summary>
    public class UpdateListFileProcessor : CommandLineProcessorBase
    {
        /// <summary>
        /// アップデートリストファイルです。
        /// </summary>
        string updateListFile;

        /// <summary>
        /// アップデートするエミッタセットファイルのリストです。
        /// </summary>
        List<string> updateEsetFiles;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="updateListFile">アップデートリストファイル</param>
        public UpdateListFileProcessor(CommandLineWorkingData workingData, string updateListFile)
            : base(workingData)
        {
            this.updateListFile = updateListFile;
        }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public override string ErrorReport { get { return string.Empty; } }

        /// <summary>
        /// 設定処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Setup()
        {
            // アップデートリストファイルをロード
            ConvertListFile convertListFile = new ConvertListFile();
            bool resLoad = convertListFile.Load(this.updateListFile);

            if (!resLoad)
            {
                return false;
            }

            this.updateEsetFiles = convertListFile.EsetPathList;

            return true;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Process()
        {
            Logger.Log("Console", LogLevels.Information, "Start updating .eset files...");

            List<string> allEsetFiles = this.updateEsetFiles;

            // FE1世代のesetファイルを列挙
            List<string> fe1EsetFiles = this.updateEsetFiles.Where(f => AncientEsetUpdater.Instance.IsFe1Eset(f)).ToList();

            // FE1世代のesetファイルをFE2世代のesetファイルにアップデート
            if (fe1EsetFiles.Count > 0)
            {
                Logger.Log("Console", LogLevels.Information, "FE1.eset files were found. Pre-updating FE1 to FE2...");

                AncientEsetUpdater.Instance.PreOpenUpdate(fe1EsetFiles, AncientEsetUpdater.EsetType.FE1);
            }

            int esetCount = allEsetFiles.Count;
            int failCount = 0;

            EmitterSetDataSerializer serializer = new EmitterSetDataSerializer();

            // FE2世代のesetファイルを、最新バージョンのesetにアップデート
            foreach (string filePath in allEsetFiles)
            {
                Logger.Log("Console", LogLevels.Information, string.Empty);
                Logger.Log("Console", LogLevels.Information, "===============================================================================");
                Logger.Log("Console", LogLevels.Information, filePath);
                Logger.Log("Console", LogLevels.Information, "===============================================================================");
                Logger.Log("Console", LogLevels.Information, "Loading emitter set from file...");

                // esetファイルをデシリアライズ
                // デシリアライズの過程でデータがアップデートされる
                string intermPath = AncientEsetUpdater.Instance.GetFe2FilePath(filePath);
                var deserializeResult = serializer.DeserializeFromFile(intermPath);

                if (deserializeResult.IsSuccess == false)
                {
                    Logger.Log("Console", LogLevels.Warning, "Failed loading emitter set from file.");

                    ++failCount;
                    continue;
                }

                Logger.Log("Console", LogLevels.Information, "Saving updated emitter set to the original file.");

                // アップデートしたデータをesetファイルにシリアライズ
                var serializeResult = serializer.SerializeToFile(deserializeResult.DataModel, filePath);

                if (serializeResult.IsSuccess == false)
                {
                    Logger.Log("Console", LogLevels.Warning, "Failed saving updated emitter set to the original file.");

                    ++failCount;
                    continue;
                }

                Logger.Log("Console", LogLevels.Information, "Successfully updated emitter set.");
            }

            Logger.Log("Console", LogLevels.Information, "Total number of emitter sets : {0}", esetCount);
            Logger.Log("Console", LogLevels.Information, "Number of failed emitter sets : {0}", failCount);

            return true;
        }

        /// <summary>
        /// 書き換えた設定などを復元します。
        /// </summary>
        public override void Cleanup()
        {
        }
    }
}
