﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using EffectMaker.BusinessLogic.DataModelOperation;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.BusinessLogic.ViewerMessages;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIDialogs.MessageDialogs;
using EffectMaker.UILogic.ViewModels;
using EmitterAndName = System.Tuple<EffectMaker.UILogic.ViewModels.EmitterViewModel, string>;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// コマンドラインで使用するバイナリコンバータです。
    /// </summary>
    public class CommandLineBinaryConverter
    {
        /// <summary>
        /// エラーログリストです。
        /// </summary>
        private readonly List<string> errorLog;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public CommandLineBinaryConverter()
        {
            this.EmitterSetList = new List<EmitterSetViewModel>();
            this.errorLog = new List<string>();
        }

        /// <summary>
        /// 出力バイナリファイルパスを取得または設定します。
        /// </summary>
        public string BinaryPath { get; set; }

        /// <summary>
        /// バイナリヘッダに追加するBinaryNameを取得または設定します。
        /// </summary>
        public string BinaryName { get; set; }

        /// <summary>
        /// プロジェクト名を取得または設定します。
        /// </summary>
        public string ProjectName { get; set; }

        /// <summary>
        /// コンバートするエミッタセットファイルパスのリストを取得または設定します。
        /// </summary>
        public List<string> EsetPathList { get; set; }

        /// <summary>
        /// コンバートリストのファイルパスを取得または設定します。
        /// </summary>
        public string ConvertListFilePath { get; set; }

        /// <summary>
        /// 入力する常駐バイナリのアセットリストファイルパスを取得または設定します。
        /// </summary>
        public string InputResidentAssetListPath { get; set; }

        /// <summary>
        /// 出力する常駐バイナリのアセットリストファイルパスを取得または設定します。
        /// </summary>
        public string OutputResidentAssetListPath { get; set; }

        /// <summary>
        /// 入力する常駐テクスチャの改行形式アセットリストファイルパスを取得または設定します。
        /// </summary>
        public string ResidentTextureListPath { get; set; }

        /// <summary>
        /// 出力シェーダバイナリのファイルパスを取得または設定します。
        /// </summary>
        public string ShaderBinaryOutputFilePath { get; set; }

        /// <summary>
        /// 出力コンピュートシェーダバイナリのファイルパスを取得または設定します。
        /// </summary>
        public string ComputeShaderBinaryOutputFilePath { get; set; }

        /// <summary>
        /// テクスチャバイナリのファイルパスを取得または設定します。
        /// </summary>
        public string TextureBinaryOutputFilePath { get; set; }

        /// <summary>
        /// G3Dプリミティブバイナリのファイルパスを取得または設定します。
        /// </summary>
        public string G3dPrimitiveBinaryOutputFilePath { get; set; }

        /// <summary>
        /// バイナリファイルを出力するかどうか取得または設定します。
        /// </summary>
        public bool OutputBinaryEnabled { get; set; }

        /// <summary>
        /// 静的解析を行うかどうか取得または設定します。
        /// </summary>
        public bool StaticProfileEnabled { get; set; }

        /// <summary>
        /// ヘッダファイルを出力するかどうか取得または設定します。
        /// </summary>
        public bool OutputHeaderEnabled { get; set; }

        /// <summary>
        /// 通常シェーダバイナリを出力するかどうか取得または設定します。
        /// </summary>
        public bool OutputShaderBinaryEnabled { get; set; }

        /// <summary>
        /// コンピュートシェーダバイナリを出力するかどうか取得または設定します。
        /// </summary>
        public bool OutputComputeShaderBinaryEnabled { get; set; }

        /// <summary>
        /// テクスチャバイナリを出力するかどうか取得または設定します。
        /// </summary>
        public bool OutputTextureBinaryEnabled { get; set; }

        /// <summary>
        /// G3Dプリミティブバイナリを出力するかどうか取得または設定します。
        /// </summary>
        public bool OutputG3dPrimitiveBinaryEnabled { get; set; }

        /// <summary>
        /// エミッタセットのファイルパスを含めるかどうか取得または設定します。
        /// </summary>
        public bool IncludeEmitterSetFilePathArray { get; set; }

        /// <summary>
        /// ヘッダファイルに書き込むエミッタセットのインデックスを記録します。
        /// </summary>
        public List<KeyValuePair<string, int>> EmitterSetIndexes { get; set; }

        /// <summary>
        /// ヘッダファイルに書き込むエミッタのインデックスを記録します。
        /// </summary>
        public List<KeyValuePair<string, int>> EmitterIndexes { get; set; }

        /// <summary>
        /// コンバートしたエミッタセットリストを取得します。
        /// </summary>
        public List<EmitterSetViewModel> EmitterSetList { get; private set; }

        /// <summary>
        /// コンバートセッションを取得します。
        /// </summary>
        public WriteBinaryDataSession Session { get; private set; }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public IEnumerable<string> ErrorLog { get { return this.errorLog; } }

        /// <summary>
        /// エラーが何も発生せずに終了できたか
        /// </summary>
        public bool IsCompletedSuccessfully { get; private set; }

        /// <summary>
        /// バイナリコンバート処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public bool Convert()
        {
            BinaryConvertProfiler.StartTotalTimer();

            OptionStore.RuntimeOptions.IsBinaryConverting = true;
            this.IsCompletedSuccessfully = false;

            // エミッタセットのロード状況をコンソールの定位置に表示するため
            // コンソールの現在のカーソル位置を取得する
            var progressCursorPos = new Point();
            var currCursorPos = new Point();
            try
            {
                progressCursorPos = new Point(Console.CursorLeft, Console.CursorTop);
            }
            catch
            {
                // スルー
            }

            // 常駐バイナリリストの読み込み
            if (File.Exists(this.InputResidentAssetListPath))
            {
                OptionStore.RuntimeOptions.IsSubBinaryConverting = true;

                var list = SerializationHelper.LoadXmlDocSerializable<ResidentAssetList>(this.InputResidentAssetListPath);
                if (list == null)
                {
                    this.errorLog.Add(string.Format("Resident List Import Error on {0}", this.InputResidentAssetListPath));
                    return false;
                }

                TextureManager.Instance.SharedGuidTable = list.TextureTable;

                // specに応じて切り替え
                if (SpecManager.CurrentSpec.BinaryHeader == "VFXB")
                {
                    // vfxの場合
                    // エミッタ形状はオリジナルプリミティブ
                    // パーティクル形状はg3dプリミティブ
                    PrimitiveManager.Instance.SharedOriginalPrimitiveGuidTable = list.OriginalPrimitiveTable;
                    PrimitiveManager.Instance.SharedG3dPrimitiveGuidTable = list.G3dPrimitiveTable;
                }
                else
                {
                    // eft2の場合
                    // 全てオリジナルプリミティブ
                    PrimitiveManager.Instance.SharedOriginalPrimitiveGuidTable =
                        list.OriginalPrimitiveTable.Concat(list.G3dPrimitiveTable).ToDictionary(x => x.Key, x => x.Value);
                }
            }

            // 改行形式の常駐バイナリリストの読み込み
            if (File.Exists(this.ResidentTextureListPath))
            {
                OptionStore.RuntimeOptions.IsSubBinaryConverting = true;

                StreamReader reader = File.OpenText(this.ResidentTextureListPath);
                while(reader.Peek() > -1)
                {
                    string line = reader.ReadLine();
                    Logger.Log(LogLevels.Information, line);

                    string extension = Path.GetExtension(line);

                    if ( extension == ".ftxb" )
                    {
                        // テクスチャの場合
                        TextureManager.Instance.AddSharedGuidTable(line);
                    }
                    else if ( extension == ".fmdb" )
                    {
                        // プリミティブの場合
                        // 下のコメントアウトを外せば、fmdbも対応可能
                        //PrimitiveManager.Instance.AddSharedGuidTable(line);
                    }
                }
                reader.Close();
            }

            BinaryConvertProfiler.StartReadingEsetTimer();
            BinaryConvertProfiler.EsetCount = this.EsetPathList.Count;

            // 全てのエミッタセットファイルを読み込む
            for (int i = 0; i < this.EsetPathList.Count; ++i)
            {
                var path = this.EsetPathList[i];

                // Output progress.
                if (IOConstants.DetailCommandLineOutputMode == false && AppData.ConsoleDisplay.IsSilentMode == false)
                {
                    // Output progress.
                    try
                    {
                        // Remember the cursor position so we can set it back
                        // after we print out the progress to console.
                        currCursorPos = new Point(Console.CursorLeft, Console.CursorTop);
                        Console.SetCursorPosition(progressCursorPos.X, progressCursorPos.Y);
                    }
                    catch
                    {
                        // If failed, just output at the current position.
                    }

                    Console.WriteLine("Loading emitter sets : {0} / {1}      ", i + 1, this.EsetPathList.Count);

                    // Set the cursor back to its original position.
                    try
                    {
                        if (currCursorPos.Y >= Console.CursorTop)
                        {
                            Console.SetCursorPosition(currCursorPos.X, currCursorPos.Y);
                        }
                    }
                    catch
                    {
                        // If failed, just output at the current position.
                    }
                }

                EmitterSetViewModel emitterSetViewModel;

                // エミッタセットファイルを読み込む
                using (new DisableMessageBoxBlock())
                {
                    var loadEmitterSetResultWithData = TheApp.RootViewModel.WorkspaceViewModel.LoadEmitterSetFile(path, true);
                    emitterSetViewModel = loadEmitterSetResultWithData.ViewModel;

                    if (emitterSetViewModel == null)
                    {
                        Logger.Log("Console", LogLevels.Error, "Eset file load failed : {0}", path);
                        continue;
                    }
                }

                // 描画パスをチェック
                // デフォルトへの差し替え、名前ベースでのID側変更はロード時に解決しているので、
                // ここでチェックすべきは、その上で名前で解決できない場合のエラーハンドリングのみ
                foreach (var emitterViewModel in emitterSetViewModel.Emitters)
                {
                    var drawPaths = OptionStore.ProjectConfig.DrawPaths;
                    var renderViewModel = emitterViewModel.EmitterBasicViewModel.EmitterBasicRenderViewModel;
                    string drawPathName = renderViewModel.DataModel.DrawPathName;

                    bool isError = false;
                    if (drawPaths.Any())
                    {
                        // バイナリコンバート時は名前で見つからなかったらエラー
                        isError = drawPaths.All(p => p.Text != drawPathName);
                    }

                    if (isError)
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Error,
                            Properties.Resources.ConsoleMsgInvalidDrawPath,
                            emitterSetViewModel.FileName,
                            emitterViewModel.Name,
                            drawPathName);

                        OptionStore.RuntimeOptions.IsBinaryConverting = false;
                        return false;
                    }
                }

                this.EmitterSetList.Add(emitterSetViewModel);
            }
            BinaryConvertProfiler.StopReadingEsetTimer();

            // バイナリコンバートを行う
            // 一度メモリ内にバイナリデータを書き出し、全ての処理が完了してからファイルに書き出す
            using (var stream = new MemoryStream())
            {
                // バイナリコンバート完了後の処理
                var completeCallback = new Action<bool, bool>((success, shaderCompileFailed) =>
                {
                    // シェーダのコンパイルに失敗したかどうかで、アプリケーションの戻り値を変える
                    this.IsCompletedSuccessfully = !shaderCompileFailed;

                    if (success == false)
                    {
                        this.errorLog.Add("Shader compiler error.");
                        return;
                    }

                    // オンラインコンパイル専用のスペックだった場合はコンバート中の警告を無視する
                    if (!SpecManager.CurrentSpec.ShaderConversionOption.TargetIsPreCompile)
                    {
                        this.IsCompletedSuccessfully = true;
                    }

                    // バイナリ出力が有効なときファイルにバイナリを書き出す
                    // 静的解析だけを行うときはバイナリを書き出さない
                    if (this.OutputBinaryEnabled)
                    {
                        using (var fs = new FileStream(this.BinaryPath, FileMode.Create, FileAccess.Write))
                        {
                            fs.Write(stream.GetBuffer(), 0, (int)stream.Length);
                        }
                    }
                });

                // バイナリコンバートセッションを作成
                this.Session = new WriteBinaryDataSession(
                    stream,
                    AssetTypes.EmitterSet,
                    this.BinaryName,
                    completeCallback,
                    false);
                Session.Profiler.EsetProfileEnabled = this.StaticProfileEnabled;
                BinaryConvertProfiler.WriteBinaryDataSessionProfiler = this.Session.Profiler;

                // 全てのエミッタセットのコンバートを行う
                for (int i = 0; i < this.EmitterSetList.Count; i++)
                {
                    EmitterSetViewModel emitterSet = this.EmitterSetList[i];

                    bool convertResult = BinaryDataHelper.WriteBinary(emitterSet.DataModel, this.Session, true);

                    if (convertResult == false)
                    {
                        this.Session.CancelSession();
                        this.errorLog.Add(string.Format("Convert error on {0}.", emitterSet.FileName));
                        OptionStore.RuntimeOptions.IsBinaryConverting = false;
                        return false;
                    }

                    // ヘッダファイル出力のオプションが指定されているときのみ実行
                    if (this.OutputHeaderEnabled == true)
                    {
                        this.ProcessEmitterSetForHeaderOutput(emitterSet, i);
                    }
                }

                // 通常シェーダを別ファイルで出力する
                if (this.OutputShaderBinaryEnabled == true)
                {
                    this.Session.ShaderBinaryOutputFilePath = PathUtility.ToAbsolutePath(this.ShaderBinaryOutputFilePath, TheApp.WorkingFolder);
                }

                // コンピュートシェーダを別ファイルで出力する
                if (this.OutputComputeShaderBinaryEnabled == true)
                {
                    this.Session.ComputeShaderBinaryOutpufFilePath = PathUtility.ToAbsolutePath(this.ComputeShaderBinaryOutputFilePath, TheApp.WorkingFolder);
                }

                // テクスチャバイナリを別ファイルで出力する
                if (this.OutputTextureBinaryEnabled == true)
                {
                    this.Session.TextureBinaryOutputFilePath = PathUtility.ToAbsolutePath(this.TextureBinaryOutputFilePath, TheApp.WorkingFolder);
                }

                // G3Dプリミティブバイナリを別ファイルで出力する
                if (this.OutputG3dPrimitiveBinaryEnabled == true)
                {
                    this.Session.G3dPrimitiveBinaryOutputFilePath = PathUtility.ToAbsolutePath(this.G3dPrimitiveBinaryOutputFilePath, TheApp.WorkingFolder);
                }

                // エミッタセットのファイルパスリストを含めるか
                if (this.IncludeEmitterSetFilePathArray == true)
                {
                    this.Session.IncludeEmitterSetFilePathArray = true;
                }

                // --resident-texture-listオプションの場合は、常駐テクスチャもnameTableに加える
                if (string.IsNullOrEmpty(this.ResidentTextureListPath) == false)
                {
                    this.Session.ForceToWriteAllTexturesToNameTable = true;
                }

                if (this.Session.EndSession() == false)
                {
                    this.errorLog.Add(string.Format("Session error on {0}", this.BinaryPath));
                    OptionStore.RuntimeOptions.IsBinaryConverting = false;
                    return false;
                }
            }

            if (!string.IsNullOrEmpty(this.OutputResidentAssetListPath))
            {
                // in,out両方指定された時のために、結合した辞書を出力する
                var texMain = TextureManager.Instance.SharedGuidTable;
                var texSub = TextureManager.Instance.GuidTable;
                var orgPrmMain = PrimitiveManager.Instance.SharedOriginalPrimitiveGuidTable;
                var g3dPrmMain = PrimitiveManager.Instance.SharedG3dPrimitiveGuidTable;

                IEnumerable<string> g3dPrimitives = null;
                IEnumerable<string> originalPrimitives = null;
                // specに応じて切り替え
                if (SpecManager.CurrentSpec.BinaryHeader == "VFXB")
                {
                    // vfxの場合
                    // エミッタ形状はオリジナルプリミティブ
                    // パーティクル形状はg3dプリミティブ
                    g3dPrimitives = this.EmitterSetList.SelectMany(eset =>
                    {
                        return eset.DataModel.EnumerateEmitterNameAndPrimitivePaths()
                            .Select(x => x.Value.ToLower());
                    });
                    originalPrimitives = this.EmitterSetList.SelectMany(eset =>
                    {
                        return eset.DataModel.EnumerateEmitterNameAndOriginalPrimitivePaths()
                            .Select(x => x.Value.ToLower());
                    });
                }
                else
                {
                    // eft2の場合
                    // 全てオリジナルプリミティブ
                    originalPrimitives = this.EmitterSetList.SelectMany(eset =>
                    {
                        return eset.DataModel.EnumerateEmitterNameAndOriginalPrimitivePaths()
                            .Union(eset.DataModel.EnumerateEmitterNameAndPrimitivePaths())
                            .Select(x => x.Value.ToLower());
                    });
                }

                // vfxもeft2も、トリミングプリミティブはオリジナル
                originalPrimitives = originalPrimitives.Concat(this.EmitterSetList.SelectMany(eset =>
                {
                    return eset.DataModel.EnumerateEmitterNameAndTrimmingTexturePaths()
                        .Select(x => x.Value.ToLower());
                }));

                // 辞書の統合
                var subG3dPrimtives = new Dictionary<string, ulong?>();
                if ( g3dPrimitives != null)
                {
                    foreach (var primitive in g3dPrimitives)
                    {
                        string key = Path.GetFileNameWithoutExtension(primitive).ToLower();
                        ulong value = PrimitiveManager.Instance.GetGuid(primitive);
                        if (subG3dPrimtives.ContainsKey(key) == false)
                        {
                            subG3dPrimtives.Add(key, value);
                        }
                    }
                }
                var subOriginalPrimitives = new Dictionary<string, ulong?>();
                if(originalPrimitives != null)
                {
                    foreach (var primitive in originalPrimitives)
                    {
                        string key = primitive.ToLower();

                        // トリミングプリミティブのときは拡張子が.ftxbになるので
                        // プリミティブファイルと被らないようにキーの拡張子を残しておく
                        if (Path.GetExtension(key) == ".ftxb")
                        {
                            key = Path.GetFileName(key);
                        }
                        else {
                            key = Path.GetFileNameWithoutExtension(key);
                        }

                        ulong value = PrimitiveManager.Instance.GetGuid(primitive);
                        if (subOriginalPrimitives.ContainsKey(key) == false)
                        {
                            subOriginalPrimitives.Add(key, value);
                        }
                    }
                }

                // Dictionaryの統合を行うローカル関数
                Func<Dictionary<string, ulong?>, Dictionary<string, ulong?>, Dictionary<string, ulong?>> combineDictionary = (dict1, dict2) =>
                {
                    Dictionary<string, ulong?> mixedDictionary = new Dictionary<string, ulong?>(dict1);

                    foreach (var item in dict2)
                    {
                        mixedDictionary[item.Key] = item.Value;
                    }

                    return mixedDictionary;
                };

                // 常駐アセットリストファイルを出力する
                var list = new ResidentAssetList
                {
                    TextureTable = combineDictionary(texMain, texSub),
                    OriginalPrimitiveTable = combineDictionary(orgPrmMain, subOriginalPrimitives),
                    G3dPrimitiveTable = combineDictionary(g3dPrmMain, subG3dPrimtives)
                };
                SerializationHelper.SaveXmlDocSerializable(list, this.OutputResidentAssetListPath);
            }

            if (OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                OptionStore.RuntimeOptions.IsSubBinaryConverting = false;
            }

            OptionStore.RuntimeOptions.IsBinaryConverting = false;

            BinaryConvertProfiler.TotalSize = this.Session.BinarySize;
            BinaryConvertProfiler.StopTotalTimer();
            BinaryConvertProfiler.OutputSummary();

            return true;
        }

        /// <summary>
        /// Process the given emitter set for header file output.
        /// </summary>
        /// <param name="eset">The view model of the emitter set.</param>
        /// <param name="index">The emitter set index.</param>
        private void ProcessEmitterSetForHeaderOutput(EmitterSetViewModel eset, int index)
        {
            // エミッタセットのヘッダファイル向けインデックス
            this.EmitterSetIndexes.Add(new KeyValuePair<string, int>(eset.FileName, index));

            // 一番上の親エミッタの逆順リストを作る
            List<EmitterViewModel> parentEmitterList = new List<EmitterViewModel>();
            foreach (EmitterViewModel parentEmitter in eset.TopEmitters)
            {
                parentEmitterList.Add(parentEmitter);
            }

            // 逆順リストへ変換
            parentEmitterList.Reverse();

            // 逆順リストをスタックにpushしていく
            Stack<EmitterAndName> emitterStack = new Stack<EmitterAndName>();
            foreach (EmitterViewModel parentEmitter in parentEmitterList)
            {
                emitterStack.Push(new EmitterAndName(
                        parentEmitter,
                        eset.FileName + "_" + parentEmitter.Name));
            }

            // エミッタセットが持っているエミッタのヘッダファイル向けインデックス
            // エミッタセット内にあるエミッタを全部出してしまう
            int emitterCounter = 0;
            while (emitterStack.Count > 0)
            {
                // エミッタをスタックからPopして、新しいエミッタのインデックスを作る
                var parentEmitter = emitterStack.Pop();

                this.EmitterIndexes.Add(
                    new KeyValuePair<string, int>(parentEmitter.Item2, emitterCounter));

                emitterCounter++;

                // popしたエミッタが持つ子エミッタの逆順リストをつくる
                List<EmitterViewModel> childEmitterList = new List<EmitterViewModel>();
                foreach (EmitterViewModel childEmitter in parentEmitter.Item1.ChildEmitters)
                {
                    childEmitterList.Add(childEmitter);
                }

                // 逆順リストへ変換
                childEmitterList.Reverse();

                // 逆順の子エミッタのリストをスタックにPushしていく
                foreach (EmitterViewModel childEmitter in childEmitterList)
                {
                    emitterStack.Push(new EmitterAndName(
                            childEmitter,
                            parentEmitter.Item2 + "_" + childEmitter.Name));
                }
            }
        }
    }
}
