﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace ShaderTyping
{
    /// <summary>
    /// Most basic definition of a shader language function: a name, 0..n parameters, and a return type.
    /// </summary>
    public class FunctionDefinition : IEquatable<FunctionDefinition>
    {
        public FunctionDefinition(
            string name,
            ShaderTypeDefinition returnType,
            params ParameterDefinition[] parameters)
        {
            if (name == null)
                throw new ArgumentNullException("name");
            if (string.IsNullOrWhiteSpace(name))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_PARAMETER, "name"), "name");
            if (returnType == null)
                throw new ArgumentNullException("returnType");

            this.Name = name;
            this.Parameters = parameters.ToArray();
            this.ReturnType = returnType;
        }

        public string Name { get; private set; }
        public ShaderTypeDefinition ReturnType { get; private set; }
        public ParameterDefinition[] Parameters { get; private set; }

        public override string ToString()
        {
            var parameters = string.Join(", ", Parameters.Select(a => a.ToString()));
            return string.Format("{0} {1}({2})", ReturnType, Name, parameters);
        }

        #region Equals

        public override bool Equals(object obj)
        {
            if (obj is FunctionDefinition)
                return Equals((FunctionDefinition)obj);

            return false;
        }

        public virtual bool Equals(FunctionDefinition other)
        {
            if ((object)other == null)
                return false;

            return Name == other.Name &&
                ReturnType == other.ReturnType &&
                Parameters.SequenceEqual(other.Parameters);
        }

        public override int GetHashCode()
        {
            return (Name +
                string.Concat(Parameters.Select(x => x.GetHashCode())) +
                ReturnType.GetHashCode()).GetHashCode();
        }

        public static bool operator ==(FunctionDefinition t1, FunctionDefinition t2)
        {
            if ((object)t1 == null)
                return (object)t2 == null;

            return t1.Equals(t2);
        }

        public static bool operator !=(FunctionDefinition t1, FunctionDefinition t2)
        {
            return !(t1 == t2);
        }

        #endregion
    }
}
